<?php
/**
 * fazer_aposta.php
 * Processa a aposta esportiva do usuário
 * Caminho: /dashboard/php/fazer_aposta.php
 */

session_start();
header('Content-Type: application/json');

// Verifica se está logado
if (!isset($_SESSION['usuario_id'])) {
    echo json_encode(['success' => false, 'message' => 'Usuário não autenticado.']);
    exit;
}

require_once '../../includes/db.php';

// Recebe dados JSON
$input = json_decode(file_get_contents('php://input'), true);

$partida_id = (int)($input['partida_id'] ?? 0);
$tipo_aposta = trim($input['tipo_aposta'] ?? '');
$odd = (float)($input['odd'] ?? 0);
$valor = (float)($input['valor'] ?? 0);
$descricao_partida = trim($input['descricao_partida'] ?? '');
$descricao_aposta = trim($input['descricao_aposta'] ?? '');
$usuario_id = $_SESSION['usuario_id'];

// Validações
if ($partida_id <= 0) {
    echo json_encode(['success' => false, 'message' => 'Partida inválida.']);
    exit;
}

if (empty($tipo_aposta)) {
    echo json_encode(['success' => false, 'message' => 'Tipo de aposta inválido.']);
    exit;
}

if ($odd <= 1) {
    echo json_encode(['success' => false, 'message' => 'Odd inválida.']);
    exit;
}

if ($valor < 1) {
    echo json_encode(['success' => false, 'message' => 'Valor mínimo para aposta é R$ 1,00.']);
    exit;
}

if ($valor > 10000) {
    echo json_encode(['success' => false, 'message' => 'Valor máximo para aposta é R$ 10.000,00.']);
    exit;
}

try {
    $pdo->beginTransaction();
    
    // Busca saldo atual do usuário
    $stmt = $pdo->prepare("SELECT bet_saldo FROM bet_usuarios WHERE id = ? FOR UPDATE");
    $stmt->execute([$usuario_id]);
    $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$usuario) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Usuário não encontrado.']);
        exit;
    }
    
    $saldo_atual = (float)$usuario['bet_saldo'];
    
    if ($valor > $saldo_atual) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Saldo insuficiente.']);
        exit;
    }
    
    // Verifica se a partida existe e está ativa
    $stmt = $pdo->prepare("SELECT id, status FROM ticker_partidas WHERE id = ? AND ativo = 1");
    $stmt->execute([$partida_id]);
    $partida = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$partida) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Partida não encontrada ou encerrada.']);
        exit;
    }
    
    if ($partida['status'] === 'finished') {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Esta partida já foi encerrada.']);
        exit;
    }
    
    // Calcula retorno potencial
    $retorno_potencial = $valor * $odd;
    
    // Desconta do saldo
    $novo_saldo = $saldo_atual - $valor;
    $stmt = $pdo->prepare("UPDATE bet_usuarios SET bet_saldo = ? WHERE id = ?");
    $stmt->execute([$novo_saldo, $usuario_id]);
    
    // Insere a aposta
    $stmt = $pdo->prepare("
        INSERT INTO apostas_esportivas 
        (usuario_id, partida_id, tipo_aposta, odd_selecionada, valor_apostado, retorno_potencial, status, descricao_partida, descricao_aposta)
        VALUES (?, ?, ?, ?, ?, ?, 'pendente', ?, ?)
    ");
    $stmt->execute([
        $usuario_id,
        $partida_id,
        $tipo_aposta,
        $odd,
        $valor,
        $retorno_potencial,
        $descricao_partida,
        $descricao_aposta
    ]);
    
    $aposta_id = $pdo->lastInsertId();
    
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Aposta realizada com sucesso!',
        'aposta_id' => $aposta_id,
        'novo_saldo' => $novo_saldo,
        'retorno_potencial' => $retorno_potencial
    ]);
    
} catch (Exception $e) {
    $pdo->rollBack();
    echo json_encode(['success' => false, 'message' => 'Erro ao processar aposta. Tente novamente.']);
}
