<?php
header('Content-Type: application/json; charset=utf-8');

/**
 * deposito.php (AJAX)
 * - POST only
 * - CSRF
 * - Logs em /logs
 * - OAuth token (2 tentativas)
 * - Gera QRCode PIX
 * - Salva em bet_transacoes
 * - ESTRUTURA CORRIGIDA BASEADA NO LARAVEL QUE FUNCIONA
 */

// ======================================================
// CONFIG LOG
// ======================================================
$DEBUG_ID = 'dep_' . date('Ymd_His') . '_' . substr(bin2hex(random_bytes(6)), 0, 12);

function log_dep($msg, $context = []) {
    global $DEBUG_ID;

    $dir = __DIR__ . '/logs';
    if (!is_dir($dir)) @mkdir($dir, 0755, true);

    $file = $dir . '/deposito-' . date('Y-m-d') . '.log';

    $line = '[' . date('Y-m-d H:i:s') . '] [' . $DEBUG_ID . '] ' . $msg;
    if (!empty($context)) {
        $line .= ' | ' . json_encode($context, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    @file_put_contents($file, $line . PHP_EOL, FILE_APPEND);
    error_log($line);
}

function json_fail($message, $extra = []) {
    global $DEBUG_ID;
    echo json_encode(array_merge([
        "success"  => false,
        "message"  => $message,
        "debug_id" => $DEBUG_ID
    ], $extra), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

function json_ok($html, $extra = []) {
    global $DEBUG_ID;
    echo json_encode(array_merge([
        "success"  => true,
        "html"     => $html,
        "debug_id" => $DEBUG_ID
    ], $extra), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// ======================================================
// SOMENTE POST
// ======================================================
$method = $_SERVER['REQUEST_METHOD'] ?? '';

log_dep("INICIO deposito.php", [
    "method" => $method,
    "uri"    => $_SERVER['REQUEST_URI'] ?? '',
    "host"   => $_SERVER['HTTP_HOST'] ?? '',
    "ip"     => $_SERVER['REMOTE_ADDR'] ?? '',
]);

if ($method !== 'POST') {
    json_fail("Método inválido.");
}

// ======================================================
// SESSION + INCLUDES
// ======================================================
session_start();

log_dep("SESSION", [
    "session_id" => session_id(),
    "has_usuario_id" => isset($_SESSION['usuario_id']) ? 1 : 0,
]);

require_once '../../includes/db.php';
require_once '../../includes/config.php';

// ======================================================
// FUNÇÕES
// ======================================================
function generateQRCodeBase64($data) {
    $url = 'https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=' . urlencode($data);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_FOLLOWLOCATION => true,
    ]);
    $imageData = curl_exec($ch);
    $httpCode  = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode === 200 && $imageData !== false) {
        return 'data:image/png;base64,' . base64_encode($imageData);
    }

    $fallbackUrl = 'https://chart.googleapis.com/chart?chs=200x200&cht=qr&chl=' . urlencode($data);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $fallbackUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
    ]);
    $imageData = curl_exec($ch);
    $httpCode  = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode === 200 && $imageData !== false) {
        return 'data:image/png;base64,' . base64_encode($imageData);
    }

    return 'data:image/svg+xml;base64,' . base64_encode(
        '<svg width="200" height="200" xmlns="http://www.w3.org/2000/svg"><rect width="200" height="200" fill="#f0f0f0"/><text x="100" y="100" text-anchor="middle" dominant-baseline="central" font-family="Arial" font-size="14">QR Code Indisponível</text></svg>'
    );
}

function valida_token_csrf($form_name) {
    $token = $_POST['csrf_token'] ?? '';
    $sess  = $_SESSION["csrf_token_$form_name"] ?? '';
    return ($token !== '' && $sess !== '' && hash_equals($sess, $token));
}

function curl_req($url, $headers, $body, $timeout = 20) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $body,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => $timeout,
    ]);
    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return [$http, $err, $resp];
}

// ======================================================
// ENTRADAS
// ======================================================
$depositoRaw = isset($_POST['deposito']) ? trim((string)$_POST['deposito']) : '';
$csrfPost    = $_POST['csrf_token'] ?? '';
$csrfSess    = $_SESSION['csrf_token_deposito'] ?? '';

log_dep("POST recebido", [
    "deposito_raw" => $depositoRaw,
    "csrf_post_12" => substr($csrfPost, 0, 12),
    "csrf_sess_12" => substr($csrfSess, 0, 12),
]);

$errors = [];

if (!valida_token_csrf('deposito')) {
    log_dep("CSRF FALHOU");
    $errors[] = "Sessão expirada. Recarregue a página e tente novamente.";
}

if ($depositoRaw === '') {
    $errors[] = "O campo depósito é obrigatório!";
}

$valorOriginal = 0.0;
if (!$errors) {
    $cleanValue = str_replace(['R$', ' ', "\xc2\xa0"], ['', '', ''], $depositoRaw);
    $cleanValue = str_replace('.', '', $cleanValue);
    $cleanValue = str_replace(',', '.', $cleanValue);

    log_dep("deposito clean", ["cleanValue" => $cleanValue]);

    if (!preg_match('/^\d+(\.\d{1,2})?$/', $cleanValue)) {
        $errors[] = "Valor não aceito!";
    } else {
        $valorOriginal = (float)$cleanValue;

        if ($valorOriginal < $ValorDeposito) {
            $errors[] = "O depósito mínimo é de R$ " . number_format($ValorDeposito, 2, ',', '.') . " reais!";
        }
        if ($valorOriginal > 10000) {
            $errors[] = "O depósito máximo é de R$ 10.000,00 reais!";
        }
    }
}

if ($errors) {
    log_dep("VALIDAÇÃO falhou", ["errors" => $errors]);
    json_fail(implode("<br>", $errors));
}

if (empty($_SESSION['usuario_id'])) {
    json_fail("Não autenticado. Recarregue a página.");
}

$usuario_id = (int)$_SESSION['usuario_id'];

// ======================================================
// PROTEÇÃO CONTRA DUPLO ENVIO (DOUBLE SUBMIT)
// ======================================================
$submitKey = 'last_deposit_submit_' . $usuario_id;
$now = microtime(true);
$lastSubmit = $_SESSION[$submitKey] ?? 0;

// Se última submissão foi há menos de 3 segundos, bloquear
if (($now - $lastSubmit) < 3) {
    log_dep("DUPLO ENVIO bloqueado", [
        "time_diff" => ($now - $lastSubmit),
        "last_submit" => $lastSubmit
    ]);
    json_fail("Aguarde alguns segundos antes de fazer outro depósito.");
}

// Registrar este envio
$_SESSION[$submitKey] = $now;

$callback_url = 'https://' . ($_SERVER['HTTP_HOST'] ?? '') . '/dashboard/php/callback_deposito.php';

log_dep("INICIO transacao", [
    "usuario_id" => $usuario_id,
    "valor"      => $valorOriginal,
    "callback"   => $callback_url
]);

// ======================================================
// CONFIG GATEWAY
// ======================================================
$GATEWAY_HOST  = 'https://api.royalbenk.com';
$AUTH_URL      = $GATEWAY_HOST . '/v2/oauth/token';
$PIX_URL       = $GATEWAY_HOST . '/v2/pix/qrcode';

// Credenciais no código (TESTE)
$ClientIdClashPay     = 'deonny_678ec148aa549';
$ClientSecretClashPay = '29616bf51bed1bed4c762c75e2ba2929bb3b3e8b628b70e66323d156bf830f7d';

// ======================================================
// PROCESSO PRINCIPAL
// ======================================================
try {
    $pdo->beginTransaction();

    // cliente
    $stmt = $pdo->prepare("SELECT bet_nome, bet_email, bet_cpf, bet_origem FROM bet_usuarios WHERE id = ?");
    $stmt->execute([$usuario_id]);
    $cliente = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$cliente) throw new Exception("Cliente não encontrado.");

    // CPF LIMPO (STRING)
    $cpfLimpo = preg_replace('/\D/', '', (string)$cliente['bet_cpf']);
    if (strlen($cpfLimpo) !== 11) {
        throw new Exception("CPF inválido no cadastro.");
    }

    // Validação adicional do nome
    $nomeCliente = trim((string)$cliente['bet_nome']);
    if (empty($nomeCliente)) {
        throw new Exception("Nome do cliente não pode estar vazio.");
    }

    // ======================================================
    // AUTH (ANTES DO PIX!)
    // ======================================================
    $basic = base64_encode(trim($ClientIdClashPay) . ':' . trim($ClientSecretClashPay));

    // Tentativa #1: usando grant_type (igual ao Laravel)
    list($authHttp, $authErr, $authResp) = curl_req(
        $AUTH_URL,
        [
            'Authorization: Basic ' . $basic,
            'Content-Type: application/x-www-form-urlencoded',
        ],
        http_build_query(['grant_type' => 'client_credentials'])
    );

    log_dep("AUTH attempt#1", [
        "http" => $authHttp,
        "curl_err" => $authErr,
        "body_300" => mb_substr((string)$authResp, 0, 300)
    ]);

    $authData = json_decode((string)$authResp, true);
    $jwtToken = $authData['access_token'] ?? null;

    // Tentativa #2: basic + json {} (fallback)
    if (!$jwtToken) {
        list($authHttp2, $authErr2, $authResp2) = curl_req(
            $AUTH_URL,
            [
                'Authorization: Basic ' . $basic,
                'Content-Type: application/json',
                'Accept: application/json',
            ],
            '{}'
        );

        log_dep("AUTH attempt#2", [
            "http" => $authHttp2,
            "curl_err" => $authErr2,
            "body_300" => mb_substr((string)$authResp2, 0, 300)
        ]);

        $authData2 = json_decode((string)$authResp2, true);
        $jwtToken = $authData2['access_token'] ?? null;

        $authHttp = $authHttp2;
        $authData = $authData2;
    }

    if (!$jwtToken) {
        $msg = $authData['message'] ?? ($authData['error'] ?? 'Falha no auth');
        throw new Exception("Erro na autenticação JWT. HTTP={$authHttp}. Msg={$msg}");
    }

    // ======================================================
    // PIX - ESTRUTURA EXATA DO LARAVEL QUE FUNCIONA
    // ======================================================
    $external_id = 'dep_' . $usuario_id . '_' . uniqid('', true);

    // ✅ ESTRUTURA IGUAL AO BSPAYTRAIT DO LARAVEL
    $payloadArr = [
        "amount"        => (float)$valorOriginal,
        "external_id"   => (string)$external_id,
        "payerQuestion" => "Pagamento referente ao serviço/produto X",
        "postbackUrl"   => $callback_url,
        
        // ✅ ATENÇÃO: No Laravel usa "name" e "document", NÃO "fullName"!
        "payer" => [
            "name"     => $nomeCliente,
            "document" => $cpfLimpo,
            "email"    => trim((string)$cliente['bet_email'])
        ]
    ];

    log_dep("PIX payload preparado (estrutura Laravel)", [
        "external_id" => $external_id,
        "amount" => $valorOriginal,
        "payer_name" => $nomeCliente,
        "payer_email" => trim((string)$cliente['bet_email']),
        "payer_document" => substr($cpfLimpo, 0, 3) . '***'
    ]);

    list($pixHttp, $pixErr, $pixResp) = curl_req(
        $PIX_URL,
        [
            'Authorization: Bearer ' . $jwtToken,
            'Content-Type: application/json',
        ],
        json_encode($payloadArr, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
    );

    log_dep("PIX response", [
        "http" => $pixHttp,
        "curl_err" => $pixErr,
        "body_500" => mb_substr((string)$pixResp, 0, 500),
        "external_id" => $external_id
    ]);

    $pixData = json_decode((string)$pixResp, true);
    $id_transacao = $pixData['transactionId'] ?? null;
    $qr_code      = $pixData['qrcode'] ?? null;

    if (!$id_transacao || !$qr_code) {
        $msg = $pixData['message'] ?? ($pixData['errorDetails']['message'] ?? 'Erro desconhecido');
        log_dep("PIX falhou", [
            "message" => $msg,
            "full_response" => $pixData
        ]);
        throw new Exception("Erro na geração do QR Code. HTTP={$pixHttp}. Msg={$msg}");
    }

    $qr_code_base64 = generateQRCodeBase64($qr_code);

    // ======================================================
    // Salvar no banco
    // ======================================================
    $origem    = $cliente['bet_origem'] ?? '';
    $ipUsuario = $_SERVER['REMOTE_ADDR'] ?? '';
    $fbc       = $_COOKIE['_fbc'] ?? '';
    $fbp       = $_COOKIE['_fbp'] ?? '';

    $insert = $pdo->prepare("
        INSERT INTO bet_transacoes
        (bet_usuario, bet_id_transacao, bet_valor, bet_tipo, bet_status, bet_data, bet_origem, bet_ip, bet_fbc, bet_fbp)
        VALUES
        (:usuario, :id_transacao, :valor, :tipo, 'Pendente', NOW(), :origem, :ip, :fbc, :fbp)
    ");

    $insert->execute([
        ':usuario'      => $usuario_id,
        ':id_transacao' => $id_transacao,
        ':valor'        => $valorOriginal,
        ':tipo'         => 'Deposito',
        ':origem'       => $origem,
        ':ip'           => $ipUsuario,
        ':fbc'          => $fbc,
        ':fbp'          => $fbp
    ]);

    $pdo->commit();

    log_dep("OK: transacao criada", [
        "transactionId" => $id_transacao,
        "external_id" => $external_id
    ]);

    // ======================================================
    // HTML simplificado para inserir no modal existente
    // ======================================================
    $html = '
    <div class="pix-gerado-container">
        <div style="text-align:center; margin-bottom:20px;">
            <i class="fas fa-check-circle" style="font-size:48px; color:#10b981;"></i>
            <h3 style="color:#fff; margin:15px 0 5px 0;">PIX Gerado com Sucesso!</h3>
            <p style="color:#9ca3af; font-size:14px;">Escaneie o QR Code ou copie o código PIX</p>
        </div>

        <div style="text-align:center; margin:20px 0; padding:15px; background:rgba(255,255,255,0.05); border-radius:12px;">
            <img src="' . $qr_code_base64 . '" alt="QR Code PIX" style="max-width:200px; height:auto; border:2px solid #10b981; border-radius:12px; padding:10px; background:#fff;" />
        </div>

        <div style="margin-bottom:20px;">
            <label style="display:block; color:#9ca3af; font-size:13px; margin-bottom:8px;">Código PIX (Copia e Cola)</label>
            <div style="display:flex; gap:8px;">
                <input type="text" id="pixCodeToCopy" value="' . htmlspecialchars($qr_code, ENT_QUOTES, 'UTF-8') . '" readonly 
                       style="flex:1; background:#252d3d; border:1px solid rgba(255,255,255,0.1); color:#fff; padding:12px; border-radius:8px; font-size:13px; font-family:monospace;">
<button id="btnCopiarPix" type="button"
  style="background:#10b981; border:none; color:#fff; padding:12px 20px; border-radius:8px; cursor:pointer;">
  <i class="fas fa-copy"></i> Copiar
</button>

            </div>
        </div>

        <div style="background:rgba(16,185,129,0.1); border:1px solid rgba(16,185,129,0.3); border-radius:10px; padding:15px; margin-bottom:20px;">
            <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:8px;">
                <span style="color:#9ca3af; font-size:13px;">Valor do PIX</span>
                <span style="color:#10b981; font-size:20px; font-weight:700;">R$ ' . number_format($valorOriginal, 2, ',', '.') . '</span>
            </div>
            <div style="display:flex; justify-content:space-between; align-items:center;">
                <span style="color:#9ca3af; font-size:12px;">ID da Transação</span>
                <span style="color:#6b7280; font-size:11px; font-family:monospace;">' . htmlspecialchars($external_id, ENT_QUOTES, 'UTF-8') . '</span>
            </div>
        </div>

        <div style="background:rgba(251,191,36,0.1); border:1px solid rgba(251,191,36,0.3); border-radius:10px; padding:12px; margin-bottom:15px; display:flex; align-items:center; gap:10px;">
            <i class="fas fa-clock" style="color:#fbbf24; font-size:18px;"></i>
            <div style="flex:1;">
                <div style="color:#fbbf24; font-weight:600; font-size:13px; margin-bottom:2px;">Aguardando Pagamento</div>
                <div style="color:#9ca3af; font-size:11px;">Assim que o pagamento for confirmado, seu saldo será atualizado automaticamente.</div>
            </div>
        </div>

        <button onclick="fecharModalDepositoComReload()" style="width:100%; background:#2d3748; border:none; color:#fff; padding:14px; border-radius:10px; cursor:pointer; font-weight:600; font-size:14px; transition:all 0.2s;">
            Fechar
        </button>

        <input type="hidden" id="pixTransactionId" value="' . htmlspecialchars($id_transacao, ENT_QUOTES, 'UTF-8') . '">
    </div>

    <script>
    function copiarCodigoPix(event) {
        event.preventDefault();
        event.stopPropagation();
        
        const input = document.getElementById("pixCodeToCopy");
        const btn = document.getElementById("btnCopiarPix");
        
        if (!input || !btn) return;
        
        // Criar um range para selecionar o texto
        input.select();
        input.setSelectionRange(0, 99999); // Para mobile
        
        // Tentar copiar usando a API moderna
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(input.value)
                .then(() => {
                    // Sucesso
                    const originalHTML = btn.innerHTML;
                    btn.innerHTML = "<i class=\"fas fa-check\"></i> Copiado!";
                    btn.style.background = "#059669";
                    
                    setTimeout(() => {
                        btn.innerHTML = originalHTML;
                        btn.style.background = "#10b981";
                    }, 2000);
                })
                .catch(err => {
                    console.error("Erro ao copiar:", err);
                    // Fallback para método antigo
                    copiarFallback(input, btn);
                });
        } else {
            // Fallback para navegadores antigos
            copiarFallback(input, btn);
        }
    }
    
    function copiarFallback(input, btn) {
        try {
            input.focus();
            input.select();
            const successful = document.execCommand("copy");
            
            if (successful) {
                const originalHTML = btn.innerHTML;
                btn.innerHTML = "<i class=\"fas fa-check\"></i> Copiado!";
                btn.style.background = "#059669";
                
                setTimeout(() => {
                    btn.innerHTML = originalHTML;
                    btn.style.background = "#10b981";
                }, 2000);
            } else {
                alert("Não foi possível copiar automaticamente. Por favor, copie manualmente.");
            }
        } catch (err) {
            console.error("Erro no fallback:", err);
            alert("Por favor, copie o código manualmente (Ctrl+C ou Cmd+C)");
        }
    }

    function fecharModalDepositoComReload() {
        location.reload();
    }

    // Verificar status do pagamento a cada 5 segundos
    (function() {
        const txid = document.getElementById("pixTransactionId")?.value;
        if (!txid) return;
        
        const intervalCheck = setInterval(function() {
            fetch("php/check_pagamento.php?txid=" + encodeURIComponent(txid) + "&t=" + Date.now())
            .then(r => r.json())
            .then(data => {
                if (data.status === "Aprovado") {
                    clearInterval(intervalCheck);
                    
                    // Mostrar alerta de sucesso
                    const container = document.querySelector(".pix-gerado-container");
                    if (container) {
                        container.innerHTML = `
                            <div style="text-align:center; padding:40px 20px;">
                                <i class="fas fa-check-circle" style="font-size:64px; color:#10b981; margin-bottom:20px;"></i>
                                <h3 style="color:#fff; margin:0 0 10px 0; font-size:24px;">Pagamento Confirmado!</h3>
                                <p style="color:#9ca3af; margin:0 0 20px 0;">Seu saldo foi atualizado com sucesso.</p>
                                <button onclick="location.reload()" style="background:#10b981; border:none; color:#fff; padding:14px 32px; border-radius:10px; cursor:pointer; font-weight:600; font-size:16px;">
                                    OK
                                </button>
                            </div>
                        `;
                    }
                    
                    setTimeout(() => location.reload(), 2000);
                }
            })
            .catch(err => console.error("Erro ao verificar pagamento:", err));
        }, 5000);
        
        // Limpar intervalo quando sair da página
        window.addEventListener("beforeunload", function() {
            clearInterval(intervalCheck);
        });
    })();
    </script>
    ';

    json_ok($html);

} catch (Exception $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();

    log_dep("ERRO EXCEPTION", [
        "msg" => $e->getMessage(),
        "trace_300" => mb_substr($e->getTraceAsString(), 0, 300)
    ]);

    json_fail("Erro ao processar depósito. Tente novamente.");
}