/**
 * TAXA DE SAQUE - INTERCEPTA AJAX EXISTENTE
 * Detecta response com status "confirmar_taxa" e abre modal
 */

(function() {
    'use strict';
    
    const TAXA_SAQUE = 30.00;
    
    console.log('[TAXA SAQUE] Script carregado - versão interceptor');
    
    // ================================================
    // INTERCEPTA O FETCH ORIGINAL
    // ================================================
    const originalFetch = window.fetch;
    window.fetch = function(...args) {
        const url = args[0];
        
        // Só intercepta requisições para retirada.php
        if (typeof url === 'string' && url.includes('retirada.php')) {
            console.log('[TAXA SAQUE] Interceptando fetch para retirada.php');
            
            return originalFetch.apply(this, args).then(response => {
                // Clona a response para poder ler o JSON
                return response.clone().json().then(data => {
                    console.log('[TAXA SAQUE] Response data:', data);
                    
                    // Verifica se precisa exibir modal de confirmação
                    if (data.status === 'confirmar_taxa' && data.data) {
                        console.log('[TAXA SAQUE] Detectado confirmar_taxa - abrindo modal');
                        
                        // Abre o modal de confirmação
                        abrirModalTaxa(data.data);
                        
                        // Retorna uma promise rejeitada para não processar mais nada
                        return Promise.reject({ 
                            intercepted: true,
                            message: 'Modal de taxa aberto'
                        });
                    }
                    
                    // Se não for confirmar_taxa, retorna a response original
                    return response;
                }).catch(err => {
                    // Se der erro ao fazer parse do JSON, retorna response original
                    if (err.intercepted) {
                        throw err;
                    }
                    return response;
                });
            });
        }
        
        // Para outras URLs, usa fetch normal
        return originalFetch.apply(this, args);
    };
    
    // ================================================
    // FUNÇÃO PARA ABRIR MODAL DE TAXA
    // ================================================
    function abrirModalTaxa(dados) {
        console.log('[TAXA SAQUE] Abrindo modal com dados:', dados);
        
        // Remove modal antigo se existir
        const modalAntigo = document.getElementById('modal-taxa-saque-overlay');
        if (modalAntigo) {
            modalAntigo.remove();
        }
        
        // Cria overlay
        const overlay = document.createElement('div');
        overlay.id = 'modal-taxa-saque-overlay';
        overlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(10px);
            z-index: 99999;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
            animation: fadeIn 0.3s ease;
        `;
        
        // Cria modal
        const modal = document.createElement('div');
        modal.style.cssText = `
            background: linear-gradient(135deg, #1e293b 0%, #0f172a 100%);
            border-radius: 20px;
            max-width: 500px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: 0 25px 50px rgba(0, 0, 0, 0.8);
            border: 1px solid rgba(255, 255, 255, 0.1);
            animation: slideUp 0.3s ease;
        `;
        
        modal.innerHTML = `
            <style>
                @keyframes fadeIn {
                    from { opacity: 0; }
                    to { opacity: 1; }
                }
                @keyframes slideUp {
                    from {
                        opacity: 0;
                        transform: translateY(30px) scale(0.95);
                    }
                    to {
                        opacity: 1;
                        transform: translateY(0) scale(1);
                    }
                }
            </style>
            
            <!-- Header -->
            <div style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); padding: 24px; text-align: center; border-radius: 20px 20px 0 0;">
                <i class="fas fa-exclamation-triangle" style="font-size: 48px; color: #fff; margin-bottom: 12px;"></i>
                <h3 style="color: #fff; margin: 0; font-size: 22px; font-weight: 700;">Confirmar Taxa de Saque</h3>
            </div>
            
            <!-- Body -->
            <div style="padding: 28px;">
                <!-- Info da taxa -->
                <div style="background: rgba(251, 191, 36, 0.1); border: 1px solid rgba(251, 191, 36, 0.3); border-radius: 12px; padding: 20px; margin-bottom: 24px;">
                    <p style="color: #fbbf24; font-weight: 600; margin: 0 0 12px 0; font-size: 15px;">
                        <i class="fas fa-info-circle"></i> Informação Importante
                    </p>
                    <p style="color: #d1d5db; margin: 0; font-size: 14px; line-height: 1.6;">
                        Para processar seu saque, será cobrada uma taxa de processamento de <strong style="color: #fbbf24;">R$ ${formatReal(dados.taxa)}</strong>.
                    </p>
                </div>
                
                <!-- Detalhes -->
                <div style="background: rgba(255, 255, 255, 0.03); border-radius: 12px; padding: 20px; margin-bottom: 24px;">
                    <div style="display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid rgba(255, 255, 255, 0.08);">
                        <span style="color: #9ca3af;">Valor do Saque</span>
                        <span style="color: #10b981; font-weight: 600;">R$ ${formatReal(dados.valor_saque)}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid rgba(255, 255, 255, 0.08);">
                        <span style="color: #9ca3af;">Taxa de Processamento</span>
                        <span style="color: #f59e0b; font-weight: 600;">R$ ${formatReal(dados.taxa)}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid rgba(255, 255, 255, 0.08);">
                        <span style="color: #9ca3af; font-weight: 600;">Valor Total a Descontar</span>
                        <span style="color: #ef4444; font-weight: 700; font-size: 18px;">R$ ${formatReal(dados.valor_total)}</span>
                    </div>
                    <div style="margin-top: 16px; padding-top: 16px; border-top: 2px solid rgba(255, 255, 255, 0.1);">
                        <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                            <span style="color: #9ca3af; font-size: 13px;">Saldo Atual</span>
                            <span style="color: #fff;">R$ ${formatReal(dados.saldo_atual)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span style="color: #9ca3af; font-size: 13px;">Saldo Após Saque</span>
                            <span style="color: #10b981; font-weight: 600;">R$ ${formatReal(dados.saldo_apos_saque)}</span>
                        </div>
                    </div>
                </div>
                
                <!-- Botões -->
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 12px;">
                    <button id="btn-cancelar-taxa" style="background: rgba(255, 255, 255, 0.05); border: 1px solid rgba(255, 255, 255, 0.1); color: #fff; padding: 14px; border-radius: 10px; cursor: pointer; font-weight: 600;">
                        <i class="fas fa-times"></i> Cancelar
                    </button>
                    <button id="btn-confirmar-taxa" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%); border: none; color: #fff; padding: 14px; border-radius: 10px; cursor: pointer; font-weight: 600; box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);">
                        <i class="fas fa-check"></i> Confirmar Saque
                    </button>
                </div>
            </div>
        `;
        
        overlay.appendChild(modal);
        document.body.appendChild(overlay);
        
        console.log('[TAXA SAQUE] Modal adicionado ao DOM');
        
        // Event listeners
        document.getElementById('btn-cancelar-taxa').onclick = () => {
            console.log('[TAXA SAQUE] Usuário cancelou');
            overlay.remove();
        };
        
        document.getElementById('btn-confirmar-taxa').onclick = () => {
            console.log('[TAXA SAQUE] Usuário confirmou');
            confirmarSaque();
            overlay.remove();
        };
        
        overlay.onclick = (e) => {
            if (e.target === overlay) {
                overlay.remove();
            }
        };
    }
    
    // ================================================
    // CONFIRMAR SAQUE COM TAXA
    // ================================================
    function confirmarSaque() {
        console.log('[TAXA SAQUE] Confirmando saque com taxa');
        
        const form = document.getElementById('formretirada');
        if (!form) {
            console.error('[TAXA SAQUE] Formulário não encontrado');
            return;
        }
        
        const btn = document.getElementById('subRetirada');
        const txtOriginal = btn ? btn.textContent : '';
        
        if (btn) {
            btn.disabled = true;
            btn.textContent = 'Processando...';
        }
        
        const formData = new FormData(form);
        formData.append('confirmar_taxa', '1');
        
        console.log('[TAXA SAQUE] Enviando com confirmar_taxa=1');
        
        // Usa o fetch original (não interceptado)
        originalFetch(form.action, {
            method: 'POST',
            headers: { 'X-Requested-With': 'XMLHttpRequest' },
            body: formData
        })
        .then(r => {
            console.log('[TAXA SAQUE] Response recebida');
            return r.json();
        })
        .then(data => {
            console.log('[TAXA SAQUE] Data:', data);
            
            if (btn) {
                btn.disabled = false;
                btn.textContent = txtOriginal;
            }
            
            if (data.status === 'alertasim') {
                mostrarAlerta(data.message, 'sucesso');
                form.reset();
                setTimeout(() => location.reload(), 2000);
            } else {
                mostrarAlerta(data.message, 'erro');
            }
        })
        .catch(err => {
            console.error('[TAXA SAQUE] Erro:', err);
            if (btn) {
                btn.disabled = false;
                btn.textContent = txtOriginal;
            }
            mostrarAlerta('Erro ao processar. Tente novamente.', 'erro');
        });
    }
    
    // ================================================
    // HELPERS
    // ================================================
    function mostrarAlerta(msg, tipo) {
        const container = document.getElementById('alerta-retirada');
        if (!container) {
            alert(msg.replace(/<[^>]*>/g, ''));
            return;
        }
        
        const cleanMsg = msg.replace(/<[^>]*>/g, '');
        const classe = tipo === 'sucesso' ? 'alertasim' : 'alertanao';
        const icon = tipo === 'sucesso' ? 'fa-check' : 'fa-times';
        
        container.innerHTML = `
            <p class="${classe}">
                ${cleanMsg}
                <span><i class="fas ${icon}"></i></span>
            </p>
        `;
        
        container.scrollIntoView({ behavior: 'smooth' });
        setTimeout(() => container.innerHTML = '', 5000);
    }
    
    function formatReal(valor) {
        return parseFloat(valor).toLocaleString('pt-BR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }
    
    console.log('[TAXA SAQUE] Interceptor ativo');
})();