<?php
/**
 * /dashboard/bingo/main.php
 * 
 * Bingo Online — página principal do jogo.
 * Compartilha autenticação com o dashboard (sessão usuario_id / cookie auth_token).
 * Conecta ao mesmo banco via ../includes/db.php e ../includes/config.php.
 */

if (session_status() === PHP_SESSION_NONE) session_start();

// ─── AUTH: mesmo fluxo do dashboard ───────────────────────────────────
if (!isset($_SESSION['usuario_id']) && isset($_COOKIE['auth_token'])) {
    require_once '../../includes/db.php';
    $token = $_COOKIE['auth_token'];
    $stmt  = $pdo->prepare("SELECT id, bet_status FROM bet_usuarios WHERE bet_token = ?");
    $stmt->execute([$token]);
    $u = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($u && $u['bet_status'] == 1) {
        $_SESSION['usuario_id'] = $u['id'];
    } else {
        setcookie("auth_token", "", time() - 3600, "/");
        header("Location: /"); exit;
    }
}
if (!isset($_SESSION['usuario_id'])) { header("Location: /"); exit; }

// ─── DEPENDÊNCIAS ─────────────────────────────────────────────────────
require_once '../../includes/db.php';
require_once '../../includes/config.php';

// ─── LOG ──────────────────────────────────────────────────────────────
$logDir = __DIR__ . '/logs';
if (!is_dir($logDir)) @mkdir($logDir, 0775, true);
function logBingo(string $msg): void {
    global $logDir;
    error_log("[".date('Y-m-d H:i:s')."] $msg\n", 3, $logDir.'/bingo_debug.log');
}

// ─── BUSCA DADOS DO USUÁRIO (bet_usuarios) ───────────────────────────
function fetchBingoUser(PDO $pdo, int $userId): array {
    $stmt = $pdo->prepare("SELECT id, bet_nome, bet_saldo FROM bet_usuarios WHERE id = ?");
    $stmt->execute([$userId]);
    $row  = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) { session_destroy(); header("Location: /"); exit; }
    $row['bet_saldo'] = (float)$row['bet_saldo'];
    return $row;
}

$user = fetchBingoUser($pdo, $_SESSION['usuario_id']);

// ─── CONFIGURAÇÕES DO JOGO ────────────────────────────────────────────
try {
    $stmt   = $pdo->prepare("SELECT * FROM configuracoes_jogo WHERE ativo = 1 LIMIT 1");
    $stmt->execute();
    $config = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
} catch (Throwable $e) { $config = []; }

$meta_arrecadacao_minima   = (float)($config['meta_arrecadacao_minima'] ?? 0.00);
$valores_cartela           = json_decode($config['valores_cartela'] ?? '["0.10","0.20","0.30","0.50","1.00","5.00","10.00","50.00"]', true);
$min_cartelas              = (int)($config['min_cartelas'] ?? 1);
$max_cartelas              = (int)($config['max_cartelas'] ?? 50);
$intervalo_jogos           = (int)($config['intervalo_jogos'] ?? 300);
$velocidade_sorteio        = (int)($config['velocidade_sorteio'] ?? 1);
$ativar_narracao           = (bool)($config['ativar_narracao'] ?? true);
$url_suporte               = $config['url_suporte'] ?? '#';

$premio_quadra_default     = (float)($config['premio_quadra_default'] ?? 20.00);
$premio_quina_default      = (float)($config['premio_quina_default'] ?? 30.00);
$premio_cartela_cheia_default = (float)($config['premio_cartela_cheia_default'] ?? 60.00);

// ─── SALA ATUAL ───────────────────────────────────────────────────────
$sala = null;
$jogo_em_andamento = false;
$jogo_terminado    = false;

// 1) Tenta sala em_andamento
$stmt = $pdo->prepare("SELECT * FROM salas_bingo WHERE status = 'em_andamento' LIMIT 1");
$stmt->execute();
$sala = $stmt->fetch(PDO::FETCH_ASSOC);
if ($sala) {
    $numeros_sorteados = json_decode($sala['numeros_sorteados'], true) ?: [];
    // Verifica se já deve finalizar (75 bolas ou status finalizado)
    if ($sala['status'] === 'finalizado' || count($numeros_sorteados) >= 75) {
        $jogo_terminado = true;
        if ($sala['status'] !== 'finalizado') {
            $pdo->prepare("UPDATE salas_bingo SET status='finalizado', fim_jogo=NOW() WHERE id=?")->execute([$sala['id']]);
            $sala['status'] = 'finalizado';
        }
    } else {
        $jogo_em_andamento = true;
    }
}

// 2) Se não há sala em_andamento, tenta aguardando
if (!$sala) {
    $stmt = $pdo->prepare("SELECT * FROM salas_bingo WHERE status = 'aguardando' LIMIT 1");
    $stmt->execute();
    $sala = $stmt->fetch(PDO::FETCH_ASSOC);
}

// 3) Cria sala se não existir
if (!$sala) {
    $numero_sorteio = rand(100000, 999999);
    $inicio = date('Y-m-d H:i:s', time() + $intervalo_jogos);
    $pdo->prepare("INSERT INTO salas_bingo (numero_sorteio, valor_cartela, premio_quadra, premio_quina, premio_cartela_cheia, inicio_previsto, arrecadacao_atual) VALUES (?, 0.10, ?, ?, ?, ?, 0.00)")
        ->execute([$numero_sorteio, $premio_quadra_default, $premio_quina_default, $premio_cartela_cheia_default, $inicio]);
    $sala_id = $pdo->lastInsertId();
    $stmt = $pdo->prepare("SELECT * FROM salas_bingo WHERE id = ?");
    $stmt->execute([$sala_id]);
    $sala = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$sala) die("Falha crítica: não foi possível criar sala de jogo. Atualize a página.");
}

// ─── LÓGICA DE INÍCIO DA SALA (aguardando → em_andamento) ────────────
$agora = time();
if ($sala && $sala['status'] === 'aguardando') {
    $pode_comecar_por_tempo = strtotime($sala['inicio_previsto']) <= $agora;
    $pode_comecar_por_meta  = false;

    if ($meta_arrecadacao_minima > 0) {
        $pode_comecar_por_meta = (float)($sala['arrecadacao_atual'] ?? 0) >= $meta_arrecadacao_minima;
    }

    $inicia = ($meta_arrecadacao_minima > 0) ? $pode_comecar_por_meta : $pode_comecar_por_tempo;

    if ($inicia) {
        $pdo->prepare("UPDATE salas_bingo SET status='em_andamento', inicio_real=NOW() WHERE id=?")->execute([$sala['id']]);
        $sala['status'] = 'em_andamento';
        $jogo_em_andamento = true;
    }
}

// ─── CARTELAS DO USUÁRIO ──────────────────────────────────────────────
$cartelas = [];
if ($sala) {
    $stmt = $pdo->prepare("SELECT * FROM cartelas_compradas WHERE usuario_id = ? AND sala_id = ? ORDER BY created_at ASC");
    $stmt->execute([$_SESSION['usuario_id'], $sala['id']]);
    $cartelas = $stmt->fetchAll(PDO::FETCH_ASSOC);
}
$cartelas_compradas_agora = count($cartelas);
$pode_comprar_mais = $cartelas_compradas_agora < $max_cartelas;

// ─── COMPRA DE CARTELAS (POST) ────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['comprar'])) {
    $preco    = (float)$_POST['preco'];
    $quantidade = (int)$_POST['quantidade'];
    $total    = $preco * $quantidade;
    $total_comprado_agora = $cartelas_compradas_agora + $quantidade;

    // Re-verificar status da sala
    $stmt = $pdo->prepare("SELECT status FROM salas_bingo WHERE id = ?");
    $stmt->execute([$sala['id']]);
    $status_atual_sala = $stmt->fetchColumn();

    if ($total_comprado_agora > $max_cartelas) {
        $_SESSION['error'] = "Você não pode comprar mais que {$max_cartelas} cartelas. Faltam " . ($max_cartelas - $cartelas_compradas_agora) . " para o limite.";
    } elseif ($status_atual_sala !== 'aguardando') {
        $_SESSION['error'] = "Não é possível comprar cartelas. O jogo está " . ($status_atual_sala === 'em_andamento' ? 'em andamento.' : 'finalizado.');
    } elseif ($quantidade >= $min_cartelas && $user['bet_saldo'] >= $total) {
        try {
            $pdo->beginTransaction();

            // Lock do usuário
            $stmt = $pdo->prepare("SELECT bet_saldo FROM bet_usuarios WHERE id = ? FOR UPDATE");
            $stmt->execute([$_SESSION['usuario_id']]);
            $saldo_atual_db = (float)$stmt->fetchColumn();

            if ($saldo_atual_db < $total) {
                throw new Exception("Saldo insuficiente no momento da compra. Saldo atual: R$ " . number_format($saldo_atual_db, 2, ',', '.'));
            }

            // Lock da sala
            $stmt = $pdo->prepare("SELECT status, arrecadacao_atual FROM salas_bingo WHERE id = ? FOR UPDATE");
            $stmt->execute([$sala['id']]);
            $sala_lock = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($sala_lock['status'] !== 'aguardando') {
                throw new Exception("Jogo iniciou ou finalizou durante a tentativa de compra.");
            }

            // Debita saldo
            $novo_saldo = $saldo_atual_db - $total;
            $pdo->prepare("UPDATE bet_usuarios SET bet_saldo = ? WHERE id = ?")->execute([$novo_saldo, $_SESSION['usuario_id']]);

            // Registra transação
            $pdo->prepare("INSERT INTO transacoes (usuario_id, tipo, valor, saldo_anterior, saldo_atual, descricao, sala_id) VALUES (?, 'compra_cartela', ?, ?, ?, ?, ?)")
                ->execute([$_SESSION['usuario_id'], $total, $saldo_atual_db, $novo_saldo, "Compra de {$quantidade} cartela(s) — Bingo", $sala['id']]);

            // Gera cartelas
            $ranges = [[1,15],[16,30],[31,45],[46,60],[61,75]];
            for ($i = 0; $i < $quantidade; $i++) {
                $numeros = [];
                foreach ($ranges as $range) {
                    $coluna = [];
                    while (count($coluna) < 5) {
                        $num = rand($range[0], $range[1]);
                        if (!in_array($num, $coluna)) $coluna[] = $num;
                    }
                    sort($coluna);
                    $numeros = array_merge($numeros, $coluna);
                }
                $pdo->prepare("INSERT INTO cartelas_compradas (usuario_id, sala_id, numeros, valor_pago) VALUES (?, ?, ?, ?)")
                    ->execute([$_SESSION['usuario_id'], $sala['id'], json_encode($numeros), $preco]);
            }

            // Atualiza arrecadação
            $nova_arrecadacao = (float)$sala_lock['arrecadacao_atual'] + $total;
            $pdo->prepare("UPDATE salas_bingo SET arrecadacao_atual = ? WHERE id = ?")->execute([$nova_arrecadacao, $sala['id']]);

            // Verifica início pela meta
            if ($meta_arrecadacao_minima > 0 && $nova_arrecadacao >= $meta_arrecadacao_minima) {
                $pdo->prepare("UPDATE salas_bingo SET status='em_andamento', inicio_real=NOW() WHERE id=?")->execute([$sala['id']]);
                $_SESSION['success'] = "Cartelas compradas! Meta de arrecadação atingida — sorteio iniciando!";
            } else {
                $_SESSION['success'] = "Cartelas compradas com sucesso! ({$quantidade} cartela(s))";
            }

            $pdo->commit();
            $user = fetchBingoUser($pdo, $_SESSION['usuario_id']); // atualiza saldo local
            header("Location: main.php"); exit;
        } catch (Exception $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            $_SESSION['error'] = "Erro ao processar compra: " . $e->getMessage();
            logBingo("Erro compra: " . $e->getMessage());
        }
    } else {
        $_SESSION['error'] = "Saldo insuficiente ou quantidade inválida.";
    }
}

// ─── DADOS COMPUTADOS ─────────────────────────────────────────────────
$numeros_sorteados = [];
if ($sala && !empty($sala['numeros_sorteados'])) {
    $numeros_sorteados = json_decode($sala['numeros_sorteados'], true) ?: [];
}

$tempo_restante = 0;
$status_arrecadacao = '';
if ($sala && $sala['status'] === 'aguardando') {
    if ($meta_arrecadacao_minima > 0) {
        $arrecadacao_atual  = (float)($sala['arrecadacao_atual'] ?? 0);
        $status_arrecadacao = number_format($arrecadacao_atual, 2, ',', '.') . ' / ' . number_format($meta_arrecadacao_minima, 2, ',', '.');
    } else {
        $tempo_restante = max(0, strtotime($sala['inicio_previsto']) - time());
    }
}

$total_premio_ganho = 0;
if ($jogo_terminado && count($cartelas) > 0) {
    $stmt = $pdo->prepare("SELECT valor FROM transacoes WHERE usuario_id = ? AND tipo = 'premio_bingo' AND sala_id = ? ORDER BY id DESC LIMIT 1");
    $stmt->execute([$_SESSION['usuario_id'], $sala['id']]);
    $total_premio_ganho = (float)($stmt->fetchColumn() ?: 0);
}

$total_gasto = array_sum(array_column($cartelas, 'valor_pago'));

// Players online simulados
if (!isset($_SESSION['players_online_base'])) $_SESSION['players_online_base'] = mt_rand(133, 350);
$players_online_base = $_SESSION['players_online_base'];

// Ganhadores recentes
$ganhadores_recentes = [];
try {
    if ($jogo_em_andamento && $sala) {
        $stmt = $pdo->prepare("
            SELECT u.bet_nome AS nome_completo, t.valor, t.descricao, t.created_at
            FROM transacoes t
            JOIN bet_usuarios u ON t.usuario_id = u.id
            WHERE t.tipo = 'premio_bingo' AND t.valor > 0 AND t.sala_id = ?
              AND (t.descricao LIKE '%Quadra%' OR t.descricao LIKE '%Quina%' OR t.descricao LIKE '%Cartela cheia%')
            ORDER BY t.created_at DESC
        ");
        $stmt->execute([$sala['id']]);
    } else {
        $stmt = $pdo->prepare("
            SELECT u.bet_nome AS nome_completo, t.valor, t.descricao, t.created_at
            FROM transacoes t
            JOIN bet_usuarios u ON t.usuario_id = u.id
            WHERE t.tipo = 'premio_bingo' AND t.valor > 0
            ORDER BY t.created_at DESC LIMIT 10
        ");
        $stmt->execute();
    }
    $ganhadores_recentes = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {}

$primeiro_nome = explode(' ', $user['bet_nome'])[0];
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Bingo Online — <?= $NomeSite ?></title>
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;900&display=swap" rel="stylesheet">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
<style>
/* ─── RESET & BASE ─────────────────────────────────────────── */
* { margin:0; padding:0; box-sizing:border-box; -webkit-tap-highlight-color:transparent; }
body {
    font-family:'Inter',Arial,sans-serif;
    background:#0f1117; color:#e2e8f0; min-height:100vh;
    -webkit-user-select:none; user-select:none;
}
:root {
    --blue:#3b82f6; --blue-hover:#2563eb;
    --gold:#fbbf24; --gold-dark:#f59e0b;
    --green:#10b981; --green-hover:#059669;
    --red:#ef4444; --red-hover:#dc2626;
    --bg:#0f1117; --card:#161922; --card-hover:#1e2330;
    --border:rgba(255,255,255,0.07); --text:#e2e8f0; --text-muted:#6b7280;
}

/* ─── HEADER MOBILE ────────────────────────────────────────── */
.bingo-top-bar {
    position:sticky; top:0; z-index:100;
    background:#111318; border-bottom:1px solid var(--border);
    padding:12px 16px; display:flex; align-items:center; justify-content:space-between;
}
.bingo-top-bar a { color:var(--blue); text-decoration:none; font-size:14px; font-weight:600; display:flex; align-items:center; gap:6px; }
.bingo-top-bar a:hover { color:var(--blue-hover); }
.bingo-top-bar .title { font-size:16px; font-weight:700; color:#fff; }
.bingo-top-bar .saldo-mini { font-size:13px; color:var(--green); font-weight:700; }

/* ─── MAIN WRAPPER ─────────────────────────────────────────── */
.bingo-wrap { max-width:900px; margin:0 auto; padding:16px; }

/* ─── TOAST ────────────────────────────────────────────────── */
.toast-wrap { position:fixed; top:16px; right:16px; z-index:9999; display:flex; flex-direction:column; gap:8px; }
.toast {
    padding:12px 18px; border-radius:10px; font-size:13px; font-weight:600;
    color:#fff; max-width:300px; transform:translateX(340px);
    transition:transform 0.3s ease; box-shadow:0 4px 16px rgba(0,0,0,0.3);
    border:1px solid transparent;
}
.toast.show { transform:translateX(0); }
.toast.success { background:var(--green); border-color:#059669; }
.toast.error   { background:var(--red); border-color:#dc2626; }
.toast.info    { background:var(--blue); border-color:#2563eb; }
.toast.user-bingo {
    background:linear-gradient(135deg,#28a745,#20c997);
    font-size:16px; padding:16px 22px;
    box-shadow:0 8px 28px rgba(40,167,69,0.5);
    animation:bounceIn 0.5s ease-out;
}
@keyframes bounceIn {
    0%   { opacity:0; transform:translateX(300px) scale(0.3); }
    50%  { opacity:1; transform:translateX(-8px) scale(1.05); }
    70%  { transform:translateX(4px) scale(0.98); }
    100% { transform:translateX(0) scale(1); }
}

/* ─── GAME HEADER CARD ─────────────────────────────────────── */
.game-header-card {
    background:var(--card); border:1px solid var(--border); border-radius:16px;
    padding:18px; margin-bottom:16px;
    display:grid; grid-template-columns:1fr auto 1fr; gap:12px; align-items:center;
}
.game-info-left { text-align:left; }
.game-info-left h3 { font-size:14px; color:var(--blue); margin-bottom:4px; font-weight:600; }
.game-info-left .countdown, .game-info-left .game-status {
    font-size:22px; font-weight:800; color:var(--gold); line-height:1.2;
}
.game-info-left .game-status { color:var(--green); }
.game-info-left .game-number { font-size:11px; color:var(--text-muted); margin-top:4px; }
.game-info-left .arrecadacao-status { font-size:16px; color:var(--blue); font-weight:700; }
.game-info-left .arrecadacao-label { font-size:10px; color:var(--text-muted); margin-top:2px; }

/* Bola central */
.last-ball-wrap { text-align:center; }
.last-ball-wrap h4 { font-size:10px; color:var(--text-muted); text-transform:uppercase; letter-spacing:0.5px; margin-bottom:6px; }
.last-ball-img { width:72px; height:72px; margin:0 auto; }
.last-ball-img img { width:100%; height:100%; object-fit:contain; filter:drop-shadow(0 2px 6px rgba(0,0,0,0.4)); }
.volume-btn {
    display:block; margin:8px auto 0; background:rgba(255,255,255,0.06); border:1px solid var(--border);
    color:var(--blue); border-radius:50%; width:32px; height:32px; cursor:pointer;
    display:flex; align-items:center; justify-content:center; margin-left:auto; margin-right:auto; margin-top:8px;
    transition:.2s;
}
.volume-btn:hover { background:rgba(255,255,255,0.1); }

/* Prêmios lado direito */
.prizes-right { text-align:right; }
.prizes-right h4 { font-size:11px; color:var(--blue); margin-bottom:6px; font-weight:600; }
.prize-row { display:flex; justify-content:space-between; font-size:12px; margin-bottom:3px; }
.prize-row .lbl { color:var(--text-muted); }
.prize-row .val { color:var(--green); font-weight:700; }
.prize-row .fee { font-size:10px; color:var(--text-muted); margin-top:6px; }

/* Players online */
.players-online-bar {
    background:var(--card); border:1px solid var(--border); border-radius:10px;
    padding:8px 16px; margin-bottom:16px; text-align:center;
    font-size:13px; color:var(--text-muted); font-weight:600;
}
.players-online-bar i { color:var(--green); margin-right:4px; }

/* Mensagem de encerramento */
.ending-message {
    background:linear-gradient(135deg,#dc3545,#e74c3c); border-radius:12px;
    padding:20px; margin-bottom:16px; text-align:center; color:#fff;
    border:2px solid rgba(255,255,255,0.2); display:none;
    animation:pulseGlow 2s infinite alternate;
}
.ending-message.winner-user { background:linear-gradient(135deg,#28a745,#20c997); }
.ending-message h3 { font-size:1.5rem; margin-bottom:8px; }
.ending-message p { font-size:14px; opacity:.9; }
@keyframes pulseGlow { 0%{box-shadow:0 6px 20px rgba(220,53,69,.4)} 100%{box-shadow:0 10px 30px rgba(220,53,69,.6)} }

/* ─── ÚLTIMAS 4 BOLAS ──────────────────────────────────────── */
.last-four-section {
    background:var(--card); border:1px solid var(--border); border-radius:14px;
    padding:14px; margin-bottom:16px;
}
.last-four-section h4 { font-size:12px; color:var(--text-muted); margin-bottom:10px; text-align:center; font-weight:600; }
.last-four-row { display:flex; justify-content:center; gap:12px; }
.ball-item { text-align:center; }
.ball-item .ball-pos { width:54px; height:54px; }
.ball-item .ball-pos img { width:100%; height:100%; object-fit:contain; }
.ball-item .ball-lbl { font-size:9px; color:var(--text-muted); margin-top:3px; }

/* ─── BOARD 1-75 ───────────────────────────────────────────── */
.board-section {
    background:var(--card); border:1px solid var(--border); border-radius:14px;
    padding:14px; margin-bottom:16px;
}
.board-section h4 { font-size:12px; color:var(--text-muted); text-align:center; margin-bottom:10px; font-weight:600; }
.numbers-board {
    display:grid; grid-template-columns:repeat(15,1fr); gap:2px;
    background:rgba(255,255,255,0.04); border-radius:8px; padding:8px;
}
.board-num {
    aspect-ratio:1; background:rgba(255,255,255,0.05); border:1px solid rgba(255,255,255,0.06);
    border-radius:4px; display:flex; align-items:center; justify-content:center;
    font-size:9px; font-weight:700; color:var(--text-muted); transition:.2s;
}
.board-num.drawn { background:var(--gold); color:#111; border-color:var(--gold-dark); }

@media(max-width:600px) {
    .numbers-board { grid-template-columns:repeat(10,1fr); }
    .board-num { font-size:8px; }
}
@media(max-width:380px) {
    .numbers-board { grid-template-columns:repeat(8,1fr); gap:1px; padding:6px; }
}

/* ─── RESULTADO (quando ganhou) ────────────────────────────── */
.result-box {
    background:var(--green); border-radius:12px; padding:18px;
    text-align:center; color:#fff; margin-bottom:16px;
}
.result-box h4 { font-size:20px; font-weight:800; margin-bottom:6px; }
.result-box .result-valor { font-size:28px; font-weight:900; color:var(--gold); }
.result-box p { font-size:13px; opacity:.85; margin-top:6px; }

/* ─── CARTELAS DO USUÁRIO ──────────────────────────────────── */
.cartelas-section { margin-bottom:16px; }
.cartelas-section h4 { font-size:13px; color:var(--text-muted); text-align:center; margin-bottom:10px; font-weight:600; }
.cartelas-grid { display:grid; grid-template-columns:repeat(auto-fill,minmax(160px,1fr)); gap:12px; }
.cartela-card {
    background:var(--card); border:1px solid var(--border); border-radius:10px;
    padding:10px; transition:.2s;
}
.cartela-card:hover { border-color:var(--blue); }
.cartela-card-header { display:flex; justify-content:space-between; font-size:10px; color:var(--text-muted); margin-bottom:6px; }
.cartela-grid-inner { display:grid; grid-template-columns:repeat(5,1fr); gap:2px; }
.cartela-num {
    aspect-ratio:1; background:rgba(255,255,255,0.05); border:1px solid rgba(255,255,255,0.06);
    border-radius:3px; display:flex; align-items:center; justify-content:center;
    font-size:8px; font-weight:700; color:var(--text-muted); transition:.2s;
}
.cartela-num.marked { background:var(--gold); color:#111; border-color:var(--gold-dark); }
.empty-cartelas {
    background:rgba(255,255,255,0.03); border:1px dashed var(--border); border-radius:10px;
    padding:28px 16px; text-align:center; color:var(--text-muted); font-size:13px;
}
.empty-cartelas i { font-size:28px; color:#333; margin-bottom:10px; display:block; }

/* ─── GANHADORES ───────────────────────────────────────────── */
.winners-section {
    background:var(--card); border:1px solid var(--border); border-radius:14px;
    padding:14px; margin-bottom:16px;
}
.winners-section h4 { font-size:12px; color:var(--blue); text-align:center; margin-bottom:10px; font-weight:600; }
.winners-list { max-height:180px; overflow-y:auto; }
.winner-item {
    display:flex; justify-content:space-between; align-items:center;
    padding:7px 10px; margin-bottom:4px; border-radius:6px;
    background:rgba(255,255,255,0.04); border-left:3px solid var(--green); font-size:11px;
}
.winner-item.in-progress { border-left-color:var(--gold-dark); background:rgba(245,158,11,0.08); }
.winner-name { font-weight:600; color:var(--text); max-width:110px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap; }
.winner-time { font-size:9px; color:var(--text-muted); }
.winner-prize { color:var(--green); font-weight:700; }
.winners-empty { text-align:center; color:var(--text-muted); font-style:italic; font-size:12px; padding:10px 0; }

/* ─── COMPRA PANEL ─────────────────────────────────────────── */
.purchase-panel {
    background:var(--card); border:1px solid var(--border); border-radius:14px;
    padding:18px; margin-bottom:16px;
}
.balance-box {
    background:var(--blue); border-radius:10px; padding:14px; text-align:center; margin-bottom:16px;
}
.balance-box .bal-amount { font-size:24px; font-weight:800; color:var(--gold); }
.balance-box .bal-label { font-size:11px; color:rgba(255,255,255,0.8); margin-top:2px; }
.balance-box .bal-spent { font-size:12px; color:rgba(255,255,255,0.65); margin-top:6px; }

.status-msg {
    padding:10px 14px; border-radius:8px; margin-bottom:14px; text-align:center;
    font-size:13px; font-weight:600; border:1px solid;
}
.status-msg.running { background:rgba(245,158,11,.1); border-color:var(--gold-dark); color:var(--gold); }
.status-msg.finished { background:rgba(16,185,129,.1); border-color:var(--green); color:var(--green); }
.status-msg.limit { background:rgba(239,68,68,.1); border-color:var(--red); color:var(--red); }
.status-msg .sm-title { font-weight:700; margin-bottom:2px; }
.status-msg .sm-sub { font-size:11px; font-weight:500; opacity:.8; }

.price-label { font-size:12px; color:var(--text-muted); font-weight:600; margin-bottom:8px; text-align:center; }
.price-grid { display:grid; grid-template-columns:repeat(4,1fr); gap:8px; margin-bottom:14px; }
.price-btn {
    padding:9px 4px; border:1px solid var(--border); border-radius:8px;
    background:rgba(255,255,255,0.04); color:var(--text-muted); font-size:11px;
    font-weight:700; cursor:pointer; transition:.2s; text-align:center;
}
.price-btn:hover:not(:disabled) { border-color:var(--blue); color:#fff; }
.price-btn.active { background:var(--blue); color:#fff; border-color:var(--blue); }
.price-btn:disabled { opacity:.4; cursor:not-allowed; }

.qty-control { display:flex; align-items:center; justify-content:center; gap:14px; margin-bottom:16px; }
.qty-btn {
    width:38px; height:38px; border:1px solid var(--blue); border-radius:50%;
    background:rgba(59,130,246,0.1); color:var(--blue); font-size:18px; font-weight:700;
    cursor:pointer; transition:.2s; display:flex; align-items:center; justify-content:center;
}
.qty-btn:hover:not(:disabled) { background:var(--blue); color:#fff; }
.qty-btn:disabled { opacity:.3; cursor:not-allowed; border-color:#333; color:#333; }
.qty-display { font-size:24px; font-weight:800; color:var(--blue); min-width:36px; text-align:center; }

.buy-btn {
    width:100%; padding:14px; background:var(--green); border:none; border-radius:10px;
    color:#fff; font-size:15px; font-weight:700; cursor:pointer; transition:.2s;
    box-shadow:0 3px 12px rgba(16,185,129,.3);
}
.buy-btn:hover:not(:disabled) { background:var(--green-hover); box-shadow:0 4px 16px rgba(16,185,129,.4); }
.buy-btn:disabled { opacity:.5; cursor:not-allowed; background:#555; box-shadow:none; }

/* ─── RESPONSIVE ───────────────────────────────────────────── */
@media(max-width:600px) {
    .game-header-card { grid-template-columns:1fr; text-align:center; }
    .game-info-left { text-align:center; }
    .prizes-right { text-align:center; }
    .prize-row { justify-content:center; gap:12px; }
    .bingo-wrap { padding:10px; }
    .last-ball-img { width:56px; height:56px; }
    .last-four-row { gap:8px; }
    .ball-item .ball-pos { width:44px; height:44px; }
}
</style>
</head>
<body>

<!-- TOP BAR -->
<div class="bingo-top-bar">
    <a href="/dashboard/"><i class="fas fa-arrow-left"></i> Dashboard</a>
    <span class="title">🎯 Bingo Online</span>
    <span class="saldo-mini">R$ <?= number_format($user['bet_saldo'], 2, ',', '.') ?></span>
</div>

<div class="bingo-wrap">

<!-- TOAST CONTAINER -->
<div class="toast-wrap" id="toastWrap">
    <?php if (isset($_SESSION['success'])): ?>
        <div class="toast show success"><?= $_SESSION['success'] ?><?php unset($_SESSION['success']); ?></div>
    <?php endif; ?>
    <?php if (isset($_SESSION['error'])): ?>
        <div class="toast show error"><?= $_SESSION['error'] ?><?php unset($_SESSION['error']); ?></div>
    <?php endif; ?>
</div>

<!-- ENDING MESSAGE -->
<div class="ending-message" id="endingMessage"></div>

<!-- GAME HEADER -->
<div class="game-header-card">
    <div class="game-info-left">
        <h3>
            <?php
            if ($jogo_terminado) echo 'Sorteio Finalizado';
            elseif ($sala && $sala['status']==='em_andamento') echo 'Sorteio em Andamento';
            else echo 'Próximo Sorteio';
            ?>
        </h3>
        <?php if ($jogo_terminado): ?>
            <div class="game-status" id="gameStatusMsg">Resultados!</div>
        <?php elseif ($sala && $sala['status']==='em_andamento'): ?>
            <div class="game-status">Sorteando...</div>
        <?php else: ?>
            <?php if ($meta_arrecadacao_minima > 0): ?>
                <div class="arrecadacao-status" id="arrecadacaoStatus"><?= $status_arrecadacao ?></div>
                <div class="arrecadacao-label" id="arrecadacaoLabel">Meta Arrecadação (R$)</div>
            <?php else: ?>
                <div class="countdown" id="countdown">--:--</div>
            <?php endif; ?>
        <?php endif; ?>
        <div class="game-number">Sorteio #<?= $sala['numero_sorteio'] ?? '000000' ?></div>
    </div>

    <div class="last-ball-wrap">
        <h4>Última Bola</h4>
        <div class="last-ball-img" id="lastBall">
            <?php
            if (!empty($numeros_sorteados)) {
                $last = end($numeros_sorteados);
                echo "<img src='assets/images/balls/{$last}.png' alt='Bola {$last}' onerror=\"this.src='assets/images/balls/interrogacao.png'\">";
            } else {
                echo "<img src='assets/images/balls/interrogacao.png' alt='Bola'>";
            }
            ?>
        </div>
        <button class="volume-btn" id="volumeToggle"><i class="fas fa-volume-up"></i></button>
    </div>

    <div class="prizes-right">
        <h4>Prêmios</h4>
        <div class="prize-row"><span class="lbl">Quadra</span><span class="val">R$ <?= number_format($sala['premio_quadra']??$premio_quadra_default,2,',','.') ?></span></div>
        <div class="prize-row"><span class="lbl">Quina</span><span class="val">R$ <?= number_format($sala['premio_quina']??$premio_quina_default,2,',','.') ?></span></div>
        <div class="prize-row"><span class="lbl">Bingo!</span><span class="val">R$ <?= number_format($sala['premio_cartela_cheia']??$premio_cartela_cheia_default,2,',','.') ?></span></div>
        <div class="prize-row fee">Taxa: R$ 0,10</div>
    </div>
</div>

<!-- PLAYERS ONLINE -->
<div class="players-online-bar">
    <i class="fas fa-users"></i> <span id="playersOnlineCount"><?= $players_online_base ?></span> Jogadores Online
</div>

<!-- ÚLTIMAS 4 BOLAS -->
<div class="last-four-section">
    <h4>Últimas 4 Bolas Sorteadas (<span id="ballCount"><?= count($numeros_sorteados) ?></span>/75)</h4>
    <div class="last-four-row">
        <?php
        $lastFour = array_slice($numeros_sorteados, -4);
        $labels = ['4ª','3ª','2ª','Última'];
        for ($i=0; $i<4; $i++) {
            $bn = $lastFour[$i] ?? null;
            echo '<div class="ball-item"><div class="ball-pos">';
            if ($bn) echo "<img src='assets/images/balls/{$bn}.png' alt='Bola {$bn}' onerror=\"this.src='assets/images/balls/interrogacao.png'\">";
            else     echo "<img src='assets/images/balls/interrogacao.png' alt='Vazio'>";
            echo "</div><div class='ball-lbl'>{$labels[$i]}</div></div>";
        }
        ?>
    </div>
</div>

<!-- BOARD 1-75 -->
<div class="board-section">
    <h4>Cartela de Bingo (1–75)</h4>
    <div class="numbers-board" id="bingoBoard">
        <?php for ($i=1; $i<=75; $i++): ?>
            <div class="board-num <?= in_array($i,$numeros_sorteados)?'drawn':'' ?>" data-number="<?=$i?>"><?=$i?></div>
        <?php endfor; ?>
    </div>
</div>

<!-- RESULTADO (se ganhou) -->
<?php if ($jogo_terminado && $total_premio_ganho > 0): ?>
<div class="result-box">
    <h4>🏆 VOCÊ GANHOU!</h4>
    <div class="result-valor">R$ <?= number_format($total_premio_ganho, 2, ',', '.') ?></div>
    <p>Seu saldo foi atualizado automaticamente.</p>
</div>
<?php endif; ?>

<!-- GANHADORES -->
<div class="winners-section">
    <h4><i class="fas fa-trophy"></i> <?= $jogo_em_andamento ? 'Ganhadores Deste Sorteio' : 'Ganhadores Recentes' ?></h4>
    <div class="winners-list" id="winnersList">
        <?php if (empty($ganhadores_recentes)): ?>
            <div class="winners-empty"><?= $jogo_em_andamento ? 'Aguardando primeiros ganhadores' : 'Nenhum ganhador ainda' ?></div>
        <?php else: ?>
            <?php foreach ($ganhadores_recentes as $g): ?>
                <?php
                $pType='';
                if (str_contains($g['descricao'],'Quadra'))      $pType='Quadra';
                elseif (str_contains($g['descricao'],'Quina'))   $pType='Quina';
                elseif (str_contains($g['descricao'],'Cartela cheia')) $pType='Bingo';
                $timeDisp = ($jogo_em_andamento && $pType) ? $pType : date('d/m H:i', strtotime($g['created_at']));
                ?>
                <div class="winner-item <?= $jogo_em_andamento?'in-progress':'' ?>">
                    <div><div class="winner-name"><?= htmlspecialchars($g['nome_completo']) ?></div><div class="winner-time"><?=$timeDisp?></div></div>
                    <div class="winner-prize">R$ <?= number_format((float)$g['valor'],2,',','.') ?></div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
</div>

<!-- CARTELAS DO USUÁRIO -->
<div class="cartelas-section">
    <h4>Suas Cartelas (<?= count($cartelas) ?>)</h4>
    <?php if (empty($cartelas)): ?>
        <div class="empty-cartelas">
            <i class="fas fa-th-large"></i>
            <p>Você ainda não possui cartelas para este sorteio.<br>Compre suas cartelas abaixo e boa sorte!</p>
        </div>
    <?php else: ?>
        <div class="cartelas-grid">
            <?php foreach ($cartelas as $idx => $cartela):
                $nums = json_decode($cartela['numeros'], true);
            ?>
                <div class="cartela-card">
                    <div class="cartela-card-header">
                        <span>Cartela <?= $idx+1 ?></span>
                        <span>R$ <?= number_format($cartela['valor_pago'],2,',','.') ?></span>
                    </div>
                    <div class="cartela-grid-inner">
                        <?php foreach ($nums as $n): ?>
                            <div class="cartela-num <?= in_array($n,$numeros_sorteados)?'marked':'' ?>" data-number="<?=$n?>"><?=$n?></div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
</div>

<!-- COMPRA PANEL -->
<div class="purchase-panel">
    <div class="balance-box">
        <div class="bal-amount">R$ <?= number_format($user['bet_saldo'],2,',','.') ?></div>
        <div class="bal-label">SALDO DISPONÍVEL</div>
        <div class="bal-spent">GASTO NESTE JOGO: R$ <?= number_format($total_gasto,2,',','.') ?> (<?= count($cartelas) ?> cartela<?= count($cartelas)!==1?'s':'' ?>)</div>
    </div>

    <?php
    $jogo_em_andamento_js = ($sala && $sala['status']==='em_andamento') || !empty($numeros_sorteados);
    $pode_comprar = !$jogo_em_andamento_js && !$jogo_terminado && $pode_comprar_mais;
    ?>

    <?php if ($jogo_em_andamento_js && !$jogo_terminado): ?>
        <div class="status-msg running"><div class="sm-title">Jogo em Andamento</div><div class="sm-sub">Não é possível comprar cartelas durante o sorteio</div></div>
    <?php endif; ?>
    <?php if ($jogo_terminado): ?>
        <div class="status-msg finished"><div class="sm-title">Sorteio Finalizado</div><div class="sm-sub">Aguarde o próximo sorteio para comprar.</div></div>
    <?php endif; ?>
    <?php if (!$pode_comprar_mais && !$jogo_em_andamento_js && !$jogo_terminado): ?>
        <div class="status-msg limit"><div class="sm-title">Limite Atingido</div><div class="sm-sub">Máximo de <?=$max_cartelas?> cartelas atingido.</div></div>
    <?php endif; ?>

    <form method="POST" id="compraForm">
        <div class="price-label">Valor da Cartela</div>
        <div class="price-grid">
            <?php foreach ($valores_cartela as $idx => $valor): ?>
                <button type="button" class="price-btn <?= $idx===0?'active':'' ?>" data-price="<?=$valor?>" <?= !$pode_comprar?'disabled':'' ?>>
                    R$ <?=$valor?>
                </button>
            <?php endforeach; ?>
        </div>

        <div class="qty-control">
            <button type="button" class="qty-btn" id="qtyMinus" <?= !$pode_comprar?'disabled':'' ?>><i class="fas fa-minus"></i></button>
            <span class="qty-display" id="qtyDisplay">1</span>
            <button type="button" class="qty-btn" id="qtyPlus" <?= !$pode_comprar?'disabled':'' ?>><i class="fas fa-plus"></i></button>
        </div>

        <input type="hidden" name="preco" id="precoInput" value="<?= $valores_cartela[0] ?>">
        <input type="hidden" name="quantidade" id="quantidadeInput" value="1">
        <button type="submit" name="comprar" class="buy-btn" id="buyBtn" <?= !$pode_comprar?'disabled':'' ?>>
            Comprar Cartelas
        </button>
    </form>
</div>

</div><!-- /bingo-wrap -->

<!-- ÁUDIOS -->
<audio id="audioQuadra" src="assets/audio/audio_quadra.mp3"></audio>
<audio id="audioQuina"  src="assets/audio/audio_quina.mp3"></audio>
<audio id="audioBingo"  src="assets/audio/audio_bingo.mp3"></audio>
<audio id="audioBola"></audio>

<script>
// ─── CONFIGURAÇÕES PASSADAS DO PHP ────────────────────────────
const userFirstName       = '<?= htmlspecialchars($primeiro_nome) ?>';
let   playersOnlineBase   = <?= $players_online_base ?>;
const salaKey             = '<?= $sala["id"] ?? 0 ?>';
let   tempoRestante       = <?= $tempo_restante ?>;
let   selectedPrice       = <?= $valores_cartela[0] ?>;
let   quantity            = 1;
let   userBalance         = <?= $user['bet_saldo'] ?>;
let   cartelas            = <?= json_encode(array_map(function($c){ $c['numeros']=json_decode($c['numeros'],true); return $c; }, $cartelas)) ?>;
let   drawnNumbers        = <?= json_encode($numeros_sorteados) ?>;
let   jogoEmAndamento     = <?= ($sala && $sala['status']==='em_andamento') ? 'true':'false' ?>;
let   jogoTerminado       = <?= $jogo_terminado ? 'true':'false' ?>;
let   cartelasCompradas   = <?= $cartelas_compradas_agora ?>;
let   maxCartelas         = <?= $max_cartelas ?>;
let   minCartelas         = <?= $min_cartelas ?>;
let   salaId              = <?= $sala['id'] ?? 'null' ?>;
let   velocidadeSorteio   = <?= $velocidade_sorteio ?>; // em segundos
let   ativarNarracao      = <?= $ativar_narracao ? 'true':'false' ?>;
const metaArrecadacaoMinima = <?= $meta_arrecadacao_minima ?>;
let   arrecadacaoAtual    = parseFloat('<?= $sala["arrecadacao_atual"] ?? "0.00" ?>');
const premioQuadra        = <?= $sala['premio_quadra']??$premio_quadra_default ?>;
const premioQuina         = <?= $sala['premio_quina']??$premio_quina_default ?>;
const premioBingo         = <?= $sala['premio_cartela_cheia']??$premio_cartela_cheia_default ?>;

// ─── ESTADO LOCAL (localStorage) ──────────────────────────────
let globalPrizesWon      = JSON.parse(localStorage.getItem(`globalPrizesWon_${salaKey}`)) || {};
let userPrizesAnnounced  = JSON.parse(localStorage.getItem(`userPrizesAnnounced_${salaKey}`)) || {};
cartelas.forEach(c => { if (!userPrizesAnnounced[c.id]) userPrizesAnnounced[c.id] = {}; });

function savePrizes() {
    localStorage.setItem(`userPrizesAnnounced_${salaKey}`, JSON.stringify(userPrizesAnnounced));
    localStorage.setItem(`globalPrizesWon_${salaKey}`, JSON.stringify(globalPrizesWon));
}
if (jogoTerminado) {
    localStorage.removeItem(`userPrizesAnnounced_${salaKey}`);
    localStorage.removeItem(`globalPrizesWon_${salaKey}`);
}

// ─── TOAST ────────────────────────────────────────────────────
function showToast(msg, type='success', dur=3000) {
    const wrap = document.getElementById('toastWrap');
    if (!wrap) return;
    const t = document.createElement('div');
    t.className = `toast ${type}`;
    t.textContent = msg;
    wrap.innerHTML = '';
    wrap.appendChild(t);
    setTimeout(() => t.classList.add('show'), 40);
    setTimeout(() => {
        t.classList.remove('show');
        setTimeout(() => t.remove(), 350);
    }, dur);
}

// Auto-dismiss initial toasts
document.querySelectorAll('.toast.show').forEach(t => {
    setTimeout(() => { t.classList.remove('show'); setTimeout(() => t.remove(), 350); }, 4000);
});

// ─── ÁUDIO ────────────────────────────────────────────────────
let isMuted = localStorage.getItem('isMuted') === 'true';

function playAudio(el) {
    if (isMuted || !ativarNarracao || !el) return;
    el.currentTime = 0;
    el.play().catch(()=>{});
}

function playBallAudio(n) {
    if (isMuted || !ativarNarracao) return;
    const a = document.getElementById('audioBola');
    if (!a) return;
    a.src = `assets/audio/bolas/${n}.mp3`;
    a.load();
    a.oncanplaythrough = () => a.play().catch(()=>{});
}

const volumeBtn  = document.getElementById('volumeToggle');
const volumeIcon = volumeBtn ? volumeBtn.querySelector('i') : null;

if (volumeIcon && isMuted) volumeIcon.classList.replace('fa-volume-up','fa-volume-mute');

if (volumeBtn) {
    volumeBtn.addEventListener('click', () => {
        isMuted = !isMuted;
        localStorage.setItem('isMuted', isMuted);
        if (volumeIcon) {
            volumeIcon.classList.toggle('fa-volume-up', !isMuted);
            volumeIcon.classList.toggle('fa-volume-mute', isMuted);
        }
        showToast(`Áudio ${isMuted?'desativado':'ativado'}`, 'info');
    });
}

// ─── PLAYERS ONLINE ───────────────────────────────────────────
let poDir = Math.random() > .5 ? 1 : -1;

function updatePlayersOnline() {
    const MIN = 133, MAX = 350;
    let v = Math.floor(Math.random()*5) + 1;

    if (playersOnlineBase + v >= MAX) poDir = -1;
    else if (playersOnlineBase - v <= MIN) poDir = 1;
    else if (Math.random() < .1) poDir *= -1;

    playersOnlineBase = Math.max(MIN, Math.min(MAX, playersOnlineBase + v*poDir));
    const el = document.getElementById('playersOnlineCount');
    if (el) el.textContent = playersOnlineBase;
}

// ─── COUNTDOWN ────────────────────────────────────────────────
function updateCountdown() {
    if (tempoRestante > 0) {
        const m = Math.floor(tempoRestante/60), s = tempoRestante%60;
        const el = document.getElementById('countdown');
        if (el) el.textContent = `${String(m).padStart(2,'0')}:${String(s).padStart(2,'0')}`;
        if (tempoRestante === 10) showToast('Apostas encerradas! Preparando sorteio.', 'error', 5000);
        tempoRestante--;
    } else {
        const el = document.getElementById('countdown');
        if (el) el.textContent = 'Sorteando...';
        jogoEmAndamento = true;
        updateButton();
    }
}

function updateArrecadacaoStatus() {
    if (metaArrecadacaoMinima <= 0) return;

    const faltam = Math.max(0, metaArrecadacaoMinima - arrecadacaoAtual);
    const el  = document.getElementById('arrecadacaoStatus');
    const lbl = document.getElementById('arrecadacaoLabel');

    if (el)  el.textContent  = `R$ ${arrecadacaoAtual.toFixed(2).replace('.', ',')} / R$ ${metaArrecadacaoMinima.toFixed(2).replace('.', ',')}`;
    if (lbl) lbl.textContent = faltam===0 ? 'META ATINGIDA! INICIANDO...' : `FALTAM R$ ${faltam.toFixed(2).replace('.', ',')}`;

    if (faltam === 0 && !jogoEmAndamento) {
        showToast('Meta atingida! Sorteio iniciando...', 'success', 5000);
        jogoEmAndamento = true;
        updateButton();
        window.location.reload();
    }
}

// ─── COMPRA UI ────────────────────────────────────────────────
function updateButton() {
    const total = selectedPrice * quantity;
    const btn   = document.getElementById('buyBtn');
    const maxA  = maxCartelas - cartelasCompradas;
    const canBuy= !jogoEmAndamento && !jogoTerminado && cartelasCompradas < maxCartelas;

    const minus = document.getElementById('qtyMinus');
    const plus  = document.getElementById('qtyPlus');

    if (minus) minus.disabled = !canBuy || quantity<=1;
    if (plus)  plus.disabled  = !canBuy || quantity>=maxA;

    document.querySelectorAll('.price-btn').forEach(b => b.disabled = !canBuy);

    if (!btn) return;

    if (!canBuy) {
        btn.disabled = true;
        if (jogoEmAndamento) btn.textContent='Jogo em Andamento';
        else if (jogoTerminado) btn.textContent='Sorteio Finalizado';
        else btn.textContent='Limite Atingido';
        return;
    }

    btn.textContent = `Comprar ${quantity} Cartela${quantity>1?'s':''} — R$ ${total.toFixed(2)}`;
    btn.disabled = total > userBalance;

    if (btn.disabled) btn.textContent = 'Saldo Insuficiente';
}

document.querySelectorAll('.price-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        document.querySelectorAll('.price-btn').forEach(b=>b.classList.remove('active'));
        this.classList.add('active');
        selectedPrice = parseFloat(this.dataset.price);
        const p = document.getElementById('precoInput');
        if (p) p.value = selectedPrice;
        updateButton();
    });
});

const qtyMinusEl = document.getElementById('qtyMinus');
const qtyPlusEl  = document.getElementById('qtyPlus');

if (qtyMinusEl) {
    qtyMinusEl.addEventListener('click', () => {
        if (quantity > minCartelas) {
            quantity--;
            const qd = document.getElementById('qtyDisplay');
            const qi = document.getElementById('quantidadeInput');
            if (qd) qd.textContent = quantity;
            if (qi) qi.value = quantity;
            updateButton();
        }
    });
}
if (qtyPlusEl) {
    qtyPlusEl.addEventListener('click', () => {
        if (quantity < (maxCartelas - cartelasCompradas)) {
            quantity++;
            const qd = document.getElementById('qtyDisplay');
            const qi = document.getElementById('quantidadeInput');
            if (qd) qd.textContent = quantity;
            if (qi) qi.value = quantity;
            updateButton();
        }
    });
}

// ─── BINGO BOARD UPDATE ──────────────────────────────────────
function updateBingoBoard(n) {
    const el = document.querySelector(`.board-num[data-number="${n}"]`);
    if (el) el.classList.add('drawn');
}

function updateLastFourBalls() {
    const last4 = drawnNumbers.slice(-4);
    const items = document.querySelectorAll('.last-four-row .ball-item');

    for (let i=0; i<4; i++) {
        const img = items[i]?.querySelector('img');
        if (!img) continue;
        const bn = last4[i] ?? null;
        img.src = bn ? `assets/images/balls/${bn}.png` : 'assets/images/balls/interrogacao.png';
        img.alt = bn ? `Bola ${bn}` : 'Vazio';
    }
}

// ─── ADICIONAR GANHADOR LOCALMENTE ────────────────────────────
function addWinnerLocally(prizeType, prizeValue) {
    const list = document.getElementById('winnersList');
    if (!list) return;

    const empty = list.querySelector('.winners-empty');
    if (empty) empty.remove();

    const valFmt = prizeValue.toLocaleString('pt-BR',{minimumFractionDigits:2});
    const item = document.createElement('div');
    item.className = 'winner-item in-progress';
    item.innerHTML = `<div><div class="winner-name">${userFirstName}</div><div class="winner-time">${prizeType}</div></div><div class="winner-prize">R$ ${valFmt}</div>`;
    list.prepend(item);
}

// ─── CHECK USER WINS ──────────────────────────────────────────
let jogoEncerrado = jogoTerminado;

function checkUserWins() {
    if (jogoEncerrado || cartelas.length===0) return;

    const drawn = new Set(drawnNumbers);
    let bingoFeito = false, changed = false;

    cartelas.forEach((cartela, idx) => {
        const nums   = cartela.numeros;
        const cId    = cartela.id;
        let   prizes = userPrizesAnnounced[cId];

        if (prizes['Cartela cheia']) return;

        // BINGO (25 acertos)
        if (nums.filter(n=>drawn.has(n)).length >= 25) {
            if (!prizes['Cartela cheia']) {
                bingoFeito = true;
                prizes['Cartela cheia'] = true;
                changed = true;
            }
        }

        // Padrões Quadra/Quina
        const patterns = [];
        for (let i=0;i<5;i++) patterns.push({id:`L${i}`, idx:[i*5,i*5+1,i*5+2,i*5+3,i*5+4]});
        for (let i=0;i<5;i++) patterns.push({id:`C${i}`, idx:[i,i+5,i+10,i+15,i+20]});
        patterns.push({id:'D0', idx:[0,6,12,18,24]});
        patterns.push({id:'D1', idx:[4,8,12,16,20]});

        patterns.forEach(p => {
            if (prizes[p.id]==='Quina' || prizes['Cartela cheia']) return;

            const matches = p.idx.filter(i=>drawn.has(nums[i])).length;

            if (matches===5 && prizes[p.id]!=='Quina' && !globalPrizesWon['Quina']) {
                playAudio(document.getElementById('audioQuina'));
                showToast(`QUINA na Cartela ${idx+1}!`, 'success', 4000);
                addWinnerLocally('Quina', premioQuina);

                prizes[p.id] = 'Quina';
                globalPrizesWon['Quina'] = true;
                changed = true;
            } else if (matches===4 && prizes[p.id]!=='Quadra' && prizes[p.id]!=='Quina' && !globalPrizesWon['Quina'] && !globalPrizesWon['Quadra']) {
                playAudio(document.getElementById('audioQuadra'));
                showToast(`QUADRA na Cartela ${idx+1}!`, 'info', 4000);
                addWinnerLocally('Quadra', premioQuadra);

                prizes[p.id] = 'Quadra';
                globalPrizesWon['Quadra'] = true;
                changed = true;
            }
        });
    });

    if (changed) savePrizes();

    // Se bingo foi feito → reporta para fechar sala
    if (bingoFeito) {
        stopPollingSorteio();
        fetch('ajax/sorteio_server.php', {
            method:'POST',
            headers:{'Content-Type':'application/json'},
            body: JSON.stringify({
                sala_id: salaId,
                bingo_feito: true,
                user_id: <?= $_SESSION['usuario_id'] ?>
            })
        })
        .then(r=>r.json())
        .then(data => {
            if (data.status==='bingo_fechado' && data.numeros_sorteados) {
                jogoEncerrado = true;
                handleGameEnd(data.numeros_sorteados);
            } else {
                // fallback: volta a buscar
                scheduleNextSorteio(velocidadeSorteio * 1000);
            }
        })
        .catch(()=> scheduleNextSorteio(velocidadeSorteio * 1000));
    }
}

// ─── RENDER FINAL STATE ───────────────────────────────────────
function renderFinalGameState(finalNumbers) {
    drawnNumbers = finalNumbers;

    const set = new Set(finalNumbers);
    document.querySelectorAll('.board-num').forEach(el => {
        const n = parseInt(el.dataset.number);
        if (set.has(n)) el.classList.add('drawn');
    });
    document.querySelectorAll('.cartela-num').forEach(el => {
        const n = parseInt(el.dataset.number);
        if (set.has(n)) el.classList.add('marked');
    });

    updateLastFourBalls();

    const bc = document.getElementById('ballCount');
    if (bc) bc.textContent = finalNumbers.length;
}

// ─── HANDLE GAME END ──────────────────────────────────────────
function handleGameEnd(finalNumbers) {
    stopPollingSorteio();
    if (poInterval) clearInterval(poInterval);

    jogoEncerrado = true;
    renderFinalGameState(finalNumbers);

    fetch('ajax/processa_premios.php', {
        method:'POST',
        headers:{'Content-Type':'application/json'},
        body: JSON.stringify({sala_id: salaId})
    })
    .then(r=>r.json())
    .then(data => {
        playAudio(document.getElementById('audioBingo'));

        const endEl = document.getElementById('endingMessage');
        const premioGanho = data.premio_ganho && parseFloat(String(data.premio_ganho).replace(',','.'))>0;

        if (data.status==='success' && data.saldo_atual) {
            userBalance = parseFloat(String(data.saldo_atual).replace(',','.'));

            const bal = document.querySelector('.bal-amount');
            const top = document.querySelector('.bingo-top-bar .saldo-mini');

            if (bal) bal.textContent = `R$ ${data.saldo}`;
            if (top) top.textContent = `R$ ${data.saldo}`;

            if (endEl) {
                if (premioGanho) {
                    endEl.className = 'ending-message winner-user';
                    endEl.innerHTML = `<h3>🏆 VOCÊ GANHOU!</h3><p>Seu prêmio de R$ ${data.premio_ganho} foi creditado!</p>`;
                    showToast(`Prêmio creditado! R$ ${data.premio_ganho}`, 'success', 8000);
                } else {
                    endEl.className = 'ending-message';
                    endEl.innerHTML = `<h3>🏆 BINGO!</h3><p>Jogo finalizado. Prêmios processados.</p>`;
                }
                endEl.style.display = 'block';
            }
        } else {
            if (endEl) {
                endEl.className = 'ending-message';
                endEl.innerHTML = `<h3>⚠️ ERRO</h3><p>Erro ao processar prêmios.</p>`;
                endEl.style.display = 'block';
            }
            showToast('Erro: '+(data.message||'falha'), 'error', 8000);
        }

        setTimeout(() => {
            showToast('Atualizando para próximo jogo...', 'info', 3000);
            setTimeout(()=>window.location.reload(), 3000);
        }, 3000);

        localStorage.removeItem(`userPrizesAnnounced_${salaKey}`);
        localStorage.removeItem(`globalPrizesWon_${salaKey}`);
    })
    .catch(() => {
        showToast('Erro de conexão. Recarregando...', 'error', 5000);
        setTimeout(()=>window.location.reload(), 5000);
    });
}

// ─── POLLING DO SORTEIO (SEM setInterval) ─────────────────────
let sorteioTimeout = null;

function stopPollingSorteio() {
    if (sorteioTimeout) clearTimeout(sorteioTimeout);
    sorteioTimeout = null;
}

function scheduleNextSorteio(ms) {
    stopPollingSorteio();
    sorteioTimeout = setTimeout(getNextBall, Math.max(120, ms));
}

function getNextBall() {
    if (!salaId || jogoEncerrado) { stopPollingSorteio(); return; }

    fetch('ajax/sorteio_server.php', {
        method:'POST',
        headers:{'Content-Type':'application/json'},
        body: JSON.stringify({sala_id: salaId})
    })
    .then(r => r.ok ? r.json() : Promise.reject('err'))
    .then(data => {
        // servidor mandou esperar (ANTI-SPAM / controle de tempo)
        if (data.status === 'aguardando_bola') {
            const w = Math.max(120, parseInt(data.wait_ms || (velocidadeSorteio * 1000), 10));
            scheduleNextSorteio(w);
            return;
        }

        // jogo fechado/finalizado
        if (data.status === 'bingo_fechado' || data.status === 'finalizado') {
            if (jogoEncerrado) return;
            jogoEncerrado = true;
            stopPollingSorteio();

            if (data.numeros_sorteados && Array.isArray(data.numeros_sorteados)) {
                handleGameEnd(data.numeros_sorteados);
            } else {
                window.location.reload();
            }
            return;
        }

        // sala aguardando (ainda não começou) → tenta de novo depois
        if (data.status === 'aguardando') {
            scheduleNextSorteio(1000);
            return;
        }

        // bola normal
        if (data.status === 'ok' && data.numero && !jogoEncerrado) {
            if (!drawnNumbers.includes(data.numero)) {
                drawnNumbers.push(data.numero);

                const lb = document.getElementById('lastBall');
                if (lb) lb.innerHTML = `<img src="assets/images/balls/${data.numero}.png" alt="Bola ${data.numero}">`;

                updateBingoBoard(data.numero);
                updateLastFourBalls();

                document.querySelectorAll('.cartela-num').forEach(el => {
                    if (parseInt(el.dataset.number) === data.numero) el.classList.add('marked');
                });

                showToast(`Número sorteado: ${data.numero}`, 'info', 1800);
                playBallAudio(data.numero);

                const bc = document.getElementById('ballCount');
                if (bc) bc.textContent = data.total_bolas || drawnNumbers.length;

                checkUserWins();
            }
        }

        // agenda próxima chamada (se o servidor mandar um tempo, usa)
        const next = Math.max(120, parseInt(data.next_in_ms || (velocidadeSorteio * 1000), 10));
        scheduleNextSorteio(next);
    })
    .catch(() => {
        // se falhar, tenta novamente sem spam
        scheduleNextSorteio(velocidadeSorteio * 1000);
    });
}

function startPollingSorteio() {
    stopPollingSorteio();
    getNextBall();
}

// ─── POLLING SALDO + GANHADORES ───────────────────────────────
function pollSaldoAndWinners() {
    const p = salaId ? `?sala_id=${salaId}` : '';
    fetch(`get_saldo.php${p}`)
    .then(r=>r.ok ? r.json() : Promise.reject('err'))
    .then(data => {
        if (data.status !== 'success') return;

        userBalance = data.saldo_float;

        const bal = document.querySelector('.bal-amount');
        const top = document.querySelector('.bingo-top-bar .saldo-mini');

        if (bal) bal.textContent = `R$ ${data.saldo}`;
        if (top) top.textContent = `R$ ${data.saldo}`;

        if (metaArrecadacaoMinima > 0 && data.arrecadacao_atual !== undefined) {
            arrecadacaoAtual = parseFloat(data.arrecadacao_atual);
            updateArrecadacaoStatus();
        }

        updateButton();

        if (!jogoEmAndamento && data.ganhadores) {
            updateWinnersPanel(data.ganhadores, data.jogo_em_andamento);
        }
    })
    .catch(()=>{});
}

function updateWinnersPanel(ganhadores, isRunning) {
    const list = document.getElementById('winnersList');
    if (!list) return;

    if (!ganhadores || ganhadores.length===0) {
        list.innerHTML = `<div class="winners-empty">${isRunning?'Aguardando primeiros ganhadores':'Nenhum ganhador ainda'}</div>`;
        return;
    }

    list.innerHTML = '';
    ganhadores.forEach(g => {
        let pType='';
        const desc = (g.descricao || '');
        if (desc.includes('Quadra')) pType='Quadra';
        else if (desc.includes('Quina')) pType='Quina';
        else if (desc.includes('Cartela cheia')) pType='Bingo';

        const td = isRunning && pType
            ? pType
            : new Date(g.created_at).toLocaleTimeString('pt-BR',{hour:'2-digit',minute:'2-digit'});

        const vf = parseFloat(g.valor).toLocaleString('pt-BR',{minimumFractionDigits:2});

        list.innerHTML += `
            <div class="winner-item ${isRunning?'in-progress':''}">
                <div>
                    <div class="winner-name">${g.nome_completo||''}</div>
                    <div class="winner-time">${td}</div>
                </div>
                <div class="winner-prize">R$ ${vf}</div>
            </div>
        `;
    });
}

// ─── INIT ─────────────────────────────────────────────────────
let poInterval;

document.addEventListener('DOMContentLoaded', () => {
    updateButton();
    updatePlayersOnline();
    poInterval = setInterval(updatePlayersOnline, 5000);

    pollSaldoAndWinners();
    setInterval(pollSaldoAndWinners, 10000);

    if (metaArrecadacaoMinima > 0) {
        updateArrecadacaoStatus();
    } else if (tempoRestante > 0 && !jogoEmAndamento) {
        updateCountdown();
        setInterval(updateCountdown, 1000);
    }

    if (jogoEmAndamento && !jogoTerminado) {
        startPollingSorteio();
    }

    if (jogoTerminado) {
        jogoEncerrado = true;
        const sm = document.getElementById('gameStatusMsg');
        if (sm) sm.textContent = 'Resultados Processados!';
    }
});
</script>

</body>
</html>
