<?php
session_start();
require_once '../includes/db.php';
require_once '../includes/config.php';

// Função de validação CSRF
function valida_token_csrf($form) {
    if (!isset($_POST['csrf_token']) || !isset($_SESSION["csrf_token_$form"])) {
        return false;
    }
    
    $token_enviado = $_POST['csrf_token'];
    $token_sessao = $_SESSION["csrf_token_$form"];
    unset($_SESSION["csrf_token_$form"]);
    
    return hash_equals($token_sessao, $token_enviado);
}

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $errors = array();
    
    error_log("DEBUG - POST recebido: " . print_r($_POST, true));
    
    $data = array(
        "nome" => htmlspecialchars(trim($_POST["nome"] ?? ''), ENT_QUOTES, 'UTF-8'),
        "email" => filter_var($_POST["email"] ?? '', FILTER_SANITIZE_EMAIL),
        "cpf" => preg_replace('/\D/', '', $_POST["cpf"] ?? ''),
        "nascimento" => $_POST["nascimento"] ?? '',
        "senha" => $_POST["senha"] ?? '',
        "confirmasenha" => $_POST["confirmasenha"] ?? '',
        "ref" => (int) ($_POST["ref"] ?? 0),
        "utm" => htmlspecialchars(trim($_POST["utm"] ?? ''), ENT_QUOTES, 'UTF-8')
    );

    // Validação CSRF
    if (!valida_token_csrf('cadastro')) {
        $errors[] = "Falha na validação de segurança. Recarregue a página.";
    }

    // Validações
    if (empty($data["nome"])) {
        $errors[] = "O campo nome é obrigatório!";
    } else if (str_word_count($data["nome"]) < 2) {
        $errors[] = "Informe o nome completo!";
    }

    if (empty($data["email"])) {
        $errors[] = "O campo email é obrigatório!";
    } else if (!filter_var($data["email"], FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Email inválido!";
    }

    if (empty($data["cpf"]) || strlen($data["cpf"]) != 11) {
        $errors[] = "CPF inválido!";
    }

    if (empty($data["nascimento"])) {
        $errors[] = "Data de nascimento é obrigatória!";
    }

    // Verifica se email já existe
    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM bet_usuarios WHERE bet_email = ?");
            $stmt->execute([$data["email"]]);
            if ($stmt->fetchColumn() > 0) {
                $errors[] = "Email já cadastrado!";
            }
        } catch (Exception $e) {
            error_log("Erro verificação email: " . $e->getMessage());
            $errors[] = "Erro interno. Tente novamente.";
        }
    }

    // Valida senha
    if (empty($errors) && empty($data["senha"])) {
        $errors[] = "O campo senha é obrigatório!";
    } else if (empty($errors) && strlen($data["senha"]) < 8) {
        $errors[] = "A senha deve ter no mínimo 8 caracteres!";
    }

    if (empty($errors) && empty($data["confirmasenha"])) {
        $errors[] = "O campo confirmar senha é obrigatório!";
    } else if (empty($errors) && $data["senha"] != $data["confirmasenha"]) {
        $errors[] = "As senhas não conferem!";
    }

    // ✅ RESPOSTA COM HTML FORMATADO
    if (count($errors) > 0) {
        error_log("DEBUG - Erros: " . print_r($errors, true));
        $response = array(
            "status" => "alertanao",
            "message" => "<div class='alerta alertanao'><i class='fa fa-times'></i> " . $errors[0] . "</div>"
        );
    } else {
        try {
            $senha_hash = password_hash($data["senha"], PASSWORD_DEFAULT);
            $token = bin2hex(random_bytes(32));
            $ip_usuario = $_SERVER['REMOTE_ADDR'] ?? '';
            
            $stmt = $pdo->prepare("INSERT INTO bet_usuarios (bet_nome, bet_email, bet_cpf, bet_nascimento, bet_senha, bet_token, bet_ip, bet_ref, bet_origem, bet_data, bet_status) VALUES (:nome, :email, :cpf, :nascimento, :senha, :token, :ip, :ref, :origem, NOW(), 1)");
            $stmt->bindParam(':nome', $data["nome"]);
            $stmt->bindParam(':email', $data["email"]);
            $stmt->bindParam(':cpf', $data["cpf"]);
            $stmt->bindParam(':nascimento', $data["nascimento"]);
            $stmt->bindParam(':senha', $senha_hash);
            $stmt->bindParam(':token', $token);
            $stmt->bindParam(':ip', $ip_usuario);
            $stmt->bindParam(':ref', $data["ref"], PDO::PARAM_INT);
            $stmt->bindParam(':origem', $data["utm"]);
            $stmt->execute();

            $novo_usuario_id = $pdo->lastInsertId();

            // Bônus de cadastro
            if (isset($ValorBonusCadastro) && $ValorBonusCadastro > 0) {
                $stmt_bonus = $pdo->prepare("INSERT INTO bet_bonus (bet_usuario, bet_bonus_tipo, bet_bonus_valor, bet_bonus_status, bet_data) VALUES (:usuario, 'Cadastro', :valor, 0, NOW())");
                $stmt_bonus->bindParam(':usuario', $novo_usuario_id, PDO::PARAM_INT);
                $stmt_bonus->bindParam(':valor', $ValorBonusCadastro);
                $stmt_bonus->execute();
            }

            // Cookie e sessão
            $cookie_options = [
                'expires' => time() + 31536000,
                'path' => '/',
                'domain' => $_SERVER['HTTP_HOST'],
                'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
                'httponly' => true,
                'samesite' => 'Strict'
            ];
            setcookie('auth_token', $token, $cookie_options);
            $_SESSION['usuario_id'] = $novo_usuario_id;

            $response = array(
                "status" => "alertasim",
                "message" => "<div class='alerta alertasim'><i class='fa fa-check'></i> Cadastro realizado com sucesso! Aguarde...</div>",
                "redirect" => "/dashboard/"
            );

            error_log("DEBUG - Cadastro OK: " . $data["email"]);

        } catch (Exception $e) {
            error_log("ERRO cadastro: " . $e->getMessage());
            
            $response = array(
                "status" => "alertanao",
                "message" => "<div class='alerta alertanao'><i class='fa fa-times'></i> Erro interno. Tente novamente.</div>"
            );
        }
    }

    // ✅ SEMPRE RETORNA JSON
    header('Content-Type: application/json');
    echo json_encode($response);
    exit;
}
?>