<?php
// Impede acesso direto via navegador (GET)
if (
    $_SERVER['REQUEST_METHOD'] !== 'POST' ||
    empty($_SERVER['HTTP_X_REQUESTED_WITH']) ||
    strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) !== 'xmlhttprequest'
) {
    header("Location: /painel/dashboard/");
    exit;
}

session_name('adm_session');
session_start();

require_once '../../../includes/db.php';

// Autenticação AJAX
require_once 'auth_ajax_adm.php';

// Função para validar CSRF dinamicamente
function valida_token_csrf($form_name) {
    $token = $_POST['csrf_token'] ?? '';
    return isset($_SESSION["csrf_token_$form_name"]) && hash_equals($_SESSION["csrf_token_$form_name"], $token);
}

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $errors = [];

    // Sanitiza entradas
    $data = [
        "bonus_usa_valor" => trim($_POST["bonus_usa_valor"] ?? ''),
        "bonus_usa_frase" => trim($_POST["bonus_usa_frase"] ?? ''),
    ];

    // CSRF -> precisa existir: $_SESSION['csrf_token_bonususa']
    if (!valida_token_csrf('bonususa')) {
        $errors[] = "Falha. Por favor, tente novamente.";
    } else {

        // ============================
        // Validação do valor do bônus
        // ============================
        if ($data["bonus_usa_valor"] === '') {
            $errors[] = "O campo valor do bônus é obrigatório!";
        } else {
            // Aceita: 750 | 750,50 | 1.750,50 | R$ 750,50
            $cleanValue = str_replace(['R$', ' ', ' '], ['', '', ''], $data["bonus_usa_valor"]);
            $cleanValue = str_replace('.', '', $cleanValue);   // remove milhares
            $cleanValue = str_replace(',', '.', $cleanValue);  // decimal BR -> EN

            if (!preg_match('/^\d+(\.\d{1,2})?$/', $cleanValue)) {
                $errors[] = "Valor do bônus não aceito!";
            } else {
                $data["bonus_usa_valor"] = (float)$cleanValue;

                if ($data["bonus_usa_valor"] < 0) {
                    $errors[] = "O valor não pode ser negativo!";
                }
            }
        }

        // ============================
        // Validação da frase
        // ============================
        if ($data["bonus_usa_frase"] === '') {
            $errors[] = "O campo frase é obrigatório!";
        } else {
            // Limita tamanho (evita frase gigante)
            if (mb_strlen($data["bonus_usa_frase"], 'UTF-8') > 255) {
                $errors[] = "A frase está muito grande (máx 255 caracteres).";
            }
        }
    }

    if (!empty($errors)) {
        $response = [
            "status" => "alertanao",
            "message" => "<p class='alertanao'>" . implode("<br>", $errors) . " <span><i class='fas fa-times'></i></span></p>"
        ];
    } else {
        try {
            $stmt = $pdo->prepare("
                UPDATE bet_adm_config
                SET bet_bonus_usa_valor = :valor,
                    bet_bonus_usa_frase = :frase
                WHERE id = 1
            ");
            $stmt->bindParam(':valor', $data['bonus_usa_valor']);
            $stmt->bindParam(':frase', $data['bonus_usa_frase']);
            $stmt->execute();

            // Regenera CSRF do bonususa
            $_SESSION['csrf_token_bonususa'] = bin2hex(random_bytes(32));

            $response = [
                "status" => "alertasim",
                "message" => "<p class='alertasim'>Bônus USA atualizado com sucesso! <span><i class='fas fa-check'></i></span></p>",
                "new_csrf" => $_SESSION['csrf_token_bonususa']
            ];
        } catch (PDOException $e) {
            $response = [
                "status" => "alertnao",
                "message" => "<p class='alertnao'>Erro ao atualizar: " . $e->getMessage() . " <span><i class='fas fa-times'></i></span></p>"
            ];
        }
    }

    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
}
?>
