<?php
session_start();
header('Content-Type: application/json');

require_once '../../../includes/db.php';

// Verifica autenticação
if (!isset($_SESSION['usuario_id']) || $_SESSION['nivel'] != 'admin') {
    echo json_encode(['status' => 'erro', 'mensagem' => 'Não autorizado']);
    exit;
}

// Verifica CSRF
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== ($_SESSION['csrf_token_bingo_tempo'] ?? '')) {
    echo json_encode(['status' => 'erro', 'mensagem' => 'Token CSRF inválido']);
    exit;
}

try {
    $intervalo_jogos = (int)($_POST['intervalo_jogos'] ?? 300);
    $velocidade_sorteio = (int)($_POST['velocidade_sorteio'] ?? 1);

    // Validações
    if ($intervalo_jogos < 30) {
        throw new Exception('Intervalo entre jogos deve ser no mínimo 30 segundos');
    }

    if ($intervalo_jogos > 3600) {
        throw new Exception('Intervalo entre jogos não pode exceder 1 hora (3600 segundos)');
    }

    if ($velocidade_sorteio < 1 || $velocidade_sorteio > 10) {
        throw new Exception('Velocidade do sorteio deve estar entre 1 e 10 segundos');
    }

    // Atualiza no banco
    $stmt = $pdo->prepare("
        UPDATE configuracoes_jogo 
        SET intervalo_jogos = ?,
            velocidade_sorteio = ?
        WHERE id = 1
    ");
    
    $stmt->execute([
        $intervalo_jogos,
        $velocidade_sorteio
    ]);

    // Gera novo token CSRF
    $_SESSION['csrf_token_bingo_tempo'] = bin2hex(random_bytes(32));

    $tempo_total = $velocidade_sorteio * 75;
    $minutos = floor($tempo_total / 60);
    $segundos = $tempo_total % 60;

    echo json_encode([
        'status' => 'sucesso',
        'mensagem' => "Configurações atualizadas! Tempo estimado por jogo: {$minutos}min {$segundos}s"
    ]);

} catch (Exception $e) {
    echo json_encode([
        'status' => 'erro',
        'mensagem' => $e->getMessage()
    ]);
}
