<?php
/**
 * sportsbook_api.php
 * API para gerenciar partidas e apostas no painel admin
 * Caminho: /painel/dashboard/php/sportsbook_api.php
 */

session_start();
header('Content-Type: application/json');

// Verifica se é admin (ajuste conforme sua autenticação)
if (!isset($_SESSION['admin_id']) && !isset($_SESSION['usuario_id'])) {
    echo json_encode(['success' => false, 'message' => 'Não autorizado']);
    exit;
}

// Caminho do db.php (ajuste se necessário)
$dbPaths = [
    '../../../includes/db.php',
    '../../includes/db.php',
    '../includes/db.php',
    'includes/db.php'
];

$dbLoaded = false;
foreach ($dbPaths as $path) {
    if (file_exists($path)) {
        require_once $path;
        $dbLoaded = true;
        break;
    }
}

if (!$dbLoaded) {
    echo json_encode(['success' => false, 'message' => 'Erro de configuração do banco de dados']);
    exit;
}

// Recebe dados JSON
$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? '';

switch ($action) {
    
    // ========================================
    // CRIAR NOVA PARTIDA
    // ========================================
    case 'criar_partida':
        $flag = trim($input['flag'] ?? '⚽');
        $liga = trim($input['liga'] ?? '');
        $time_casa = trim($input['time_casa'] ?? '');
        $time_fora = trim($input['time_fora'] ?? '');
        $placar_casa = (int)($input['placar_casa'] ?? 0);
        $placar_fora = (int)($input['placar_fora'] ?? 0);
        $tempo = trim($input['tempo'] ?? 'AO VIVO');
        $status = $input['status'] ?? 'live';
        $odd_label_1 = trim($input['odd_label_1'] ?? 'Mais de 2.5');
        $odd_valor_1 = (float)($input['odd_valor_1'] ?? 1.85);
        $odd_label_2 = trim($input['odd_label_2'] ?? 'Menos de 2.5');
        $odd_valor_2 = (float)($input['odd_valor_2'] ?? 1.95);
        $ordem = (int)($input['ordem'] ?? 0);
        $ativo = (int)($input['ativo'] ?? 1);
        
        if (empty($liga) || empty($time_casa) || empty($time_fora)) {
            echo json_encode(['success' => false, 'message' => 'Preencha liga e times']);
            exit;
        }
        
        if (!in_array($status, ['live', 'upcoming', 'finished'])) {
            $status = 'live';
        }
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO ticker_partidas 
                (flag, liga, time_casa, time_fora, placar_casa, placar_fora, tempo, status, odd_label_1, odd_valor_1, odd_label_2, odd_valor_2, ordem, ativo)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $flag, $liga, $time_casa, $time_fora, $placar_casa, $placar_fora,
                $tempo, $status, $odd_label_1, $odd_valor_1, $odd_label_2, $odd_valor_2, $ordem, $ativo
            ]);
            
            echo json_encode(['success' => true, 'message' => 'Partida cadastrada com sucesso!', 'id' => $pdo->lastInsertId()]);
        } catch (PDOException $e) {
            echo json_encode(['success' => false, 'message' => 'Erro ao cadastrar: ' . $e->getMessage()]);
        }
        break;
    
    // ========================================
    // EDITAR PARTIDA
    // ========================================
    case 'editar_partida':
        $id = (int)($input['id'] ?? 0);
        
        if ($id <= 0) {
            echo json_encode(['success' => false, 'message' => 'ID inválido']);
            exit;
        }
        
        $flag = trim($input['flag'] ?? '⚽');
        $liga = trim($input['liga'] ?? '');
        $time_casa = trim($input['time_casa'] ?? '');
        $time_fora = trim($input['time_fora'] ?? '');
        $placar_casa = (int)($input['placar_casa'] ?? 0);
        $placar_fora = (int)($input['placar_fora'] ?? 0);
        $tempo = trim($input['tempo'] ?? '');
        $status = $input['status'] ?? 'live';
        $odd_label_1 = trim($input['odd_label_1'] ?? '');
        $odd_valor_1 = (float)($input['odd_valor_1'] ?? 1.50);
        $odd_label_2 = trim($input['odd_label_2'] ?? '');
        $odd_valor_2 = (float)($input['odd_valor_2'] ?? 2.00);
        $ordem = (int)($input['ordem'] ?? 0);
        $ativo = (int)($input['ativo'] ?? 1);
        
        try {
            $stmt = $pdo->prepare("
                UPDATE ticker_partidas SET
                    flag = ?, liga = ?, time_casa = ?, time_fora = ?,
                    placar_casa = ?, placar_fora = ?, tempo = ?, status = ?,
                    odd_label_1 = ?, odd_valor_1 = ?, odd_label_2 = ?, odd_valor_2 = ?,
                    ordem = ?, ativo = ?
                WHERE id = ?
            ");
            $stmt->execute([
                $flag, $liga, $time_casa, $time_fora, $placar_casa, $placar_fora,
                $tempo, $status, $odd_label_1, $odd_valor_1, $odd_label_2, $odd_valor_2,
                $ordem, $ativo, $id
            ]);
            
            echo json_encode(['success' => true, 'message' => 'Partida atualizada!']);
        } catch (PDOException $e) {
            echo json_encode(['success' => false, 'message' => 'Erro ao atualizar: ' . $e->getMessage()]);
        }
        break;
    
    // ========================================
    // DELETAR PARTIDA
    // ========================================
    case 'deletar_partida':
        $id = (int)($input['id'] ?? 0);
        
        if ($id <= 0) {
            echo json_encode(['success' => false, 'message' => 'ID inválido']);
            exit;
        }
        
        try {
            // Verifica se há apostas vinculadas
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM apostas_esportivas WHERE partida_id = ?");
            $stmt->execute([$id]);
            $count = (int)$stmt->fetchColumn();
            
            if ($count > 0) {
                echo json_encode(['success' => false, 'message' => 'Não é possível excluir: existem ' . $count . ' apostas vinculadas']);
                exit;
            }
            
            $stmt = $pdo->prepare("DELETE FROM ticker_partidas WHERE id = ?");
            $stmt->execute([$id]);
            
            echo json_encode(['success' => true, 'message' => 'Partida excluída!']);
        } catch (PDOException $e) {
            echo json_encode(['success' => false, 'message' => 'Erro ao excluir: ' . $e->getMessage()]);
        }
        break;
    
    // ========================================
    // LIQUIDAR APOSTA (GANHOU/PERDEU)
    // ========================================
    case 'liquidar_aposta':
        $id = (int)($input['id'] ?? 0);
        $resultado = $input['resultado'] ?? '';
        
        if ($id <= 0) {
            echo json_encode(['success' => false, 'message' => 'ID inválido']);
            exit;
        }
        
        if (!in_array($resultado, ['ganhou', 'perdeu'])) {
            echo json_encode(['success' => false, 'message' => 'Resultado inválido']);
            exit;
        }
        
        try {
            $pdo->beginTransaction();
            
            // Busca dados da aposta
            $stmt = $pdo->prepare("SELECT * FROM apostas_esportivas WHERE id = ? AND status = 'pendente'");
            $stmt->execute([$id]);
            $aposta = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$aposta) {
                $pdo->rollBack();
                echo json_encode(['success' => false, 'message' => 'Aposta não encontrada ou já liquidada']);
                exit;
            }
            
            // Atualiza status da aposta
            $stmt = $pdo->prepare("UPDATE apostas_esportivas SET status = ? WHERE id = ?");
            $stmt->execute([$resultado, $id]);
            
            // Se ganhou, credita o retorno ao usuário
            if ($resultado === 'ganhou') {
                $retorno = (float)$aposta['retorno_potencial'];
                $usuario_id = (int)$aposta['usuario_id'];
                
                $stmt = $pdo->prepare("UPDATE bet_usuarios SET bet_saldo = bet_saldo + ? WHERE id = ?");
                $stmt->execute([$retorno, $usuario_id]);
                
                $pdo->commit();
                echo json_encode([
                    'success' => true, 
                    'message' => 'Aposta marcada como GANHOU! R$ ' . number_format($retorno, 2, ',', '.') . ' creditado ao usuário.'
                ]);
            } else {
                $pdo->commit();
                echo json_encode(['success' => true, 'message' => 'Aposta marcada como PERDEU.']);
            }
            
        } catch (PDOException $e) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => 'Erro ao liquidar: ' . $e->getMessage()]);
        }
        break;
    
    // ========================================
    // LISTAR PARTIDAS
    // ========================================
    case 'listar_partidas':
        try {
            $stmt = $pdo->prepare("SELECT * FROM ticker_partidas ORDER BY ordem ASC, id DESC");
            $stmt->execute();
            $partidas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['success' => true, 'partidas' => $partidas]);
        } catch (PDOException $e) {
            echo json_encode(['success' => false, 'message' => 'Erro ao listar']);
        }
        break;
    
    // ========================================
    // LISTAR APOSTAS
    // ========================================
    case 'listar_apostas':
        $status_filtro = $input['status'] ?? 'todas';
        
        try {
            $sql = "
                SELECT a.*, u.bet_nome, u.bet_email
                FROM apostas_esportivas a
                LEFT JOIN bet_usuarios u ON a.usuario_id = u.id
            ";
            
            if ($status_filtro !== 'todas' && in_array($status_filtro, ['pendente', 'ganhou', 'perdeu'])) {
                $sql .= " WHERE a.status = ?";
                $stmt = $pdo->prepare($sql . " ORDER BY a.created_at DESC LIMIT 100");
                $stmt->execute([$status_filtro]);
            } else {
                $stmt = $pdo->prepare($sql . " ORDER BY a.created_at DESC LIMIT 100");
                $stmt->execute();
            }
            
            $apostas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['success' => true, 'apostas' => $apostas]);
        } catch (PDOException $e) {
            echo json_encode(['success' => false, 'message' => 'Erro ao listar apostas']);
        }
        break;
    
    // ========================================
    // AÇÃO DESCONHECIDA
    // ========================================
    default:
        echo json_encode(['success' => false, 'message' => 'Ação não reconhecida: ' . $action]);
        break;
}
