<?php
header('Content-Type: application/json; charset=utf-8');

function json_exit($arr, $code = 200) {
  http_response_code($code);
  echo json_encode($arr, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

// ✅ Só aceita POST (sem redirect)
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_exit(["status" => "alertanao", "message" => "<p class='alertanao'>Método inválido. <span><i class='fas fa-times'></i></span></p>"], 405);
}

// Sessão
session_name('adm_session');
session_start();

// DB + Auth
require_once '../../../includes/db.php';
require_once 'auth_ajax_adm.php';

// ✅ CSRF
function valida_token_csrf($form_name) {
  $token = $_POST['csrf_token'] ?? '';
  $sess  = $_SESSION["csrf_token_$form_name"] ?? '';
  return ($token && $sess && hash_equals($sess, $token));
}

$errors = [];

// Valida CSRF
if (!valida_token_csrf('logo')) {
  $errors[] = "Falha (CSRF). Recarregue a página e tente novamente.";
}

// Valida arquivo
if (!isset($_FILES['logo']) || ($_FILES['logo']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
  $errors[] = "Nenhum arquivo foi enviado!";
}

if (!empty($errors)) {
  json_exit([
    "status"  => "alertanao",
    "message" => "<p class='alertanao'>".implode("<br>", $errors)." <span><i class='fas fa-times'></i></span></p>"
  ], 400);
}

// ✅ Caminho absoluto seguro
$uploadDir = realpath(__DIR__ . '/../../../imagens');
if ($uploadDir === false) {
  // tenta criar
  $tryCreate = __DIR__ . '/../../../imagens';
  if (!is_dir($tryCreate)) @mkdir($tryCreate, 0775, true);
  $uploadDir = realpath($tryCreate);
}

if ($uploadDir === false || !is_dir($uploadDir) || !is_writable($uploadDir)) {
  json_exit([
    "status"  => "alertanao",
    "message" => "<p class='alertanao'>Pasta de upload inválida/sem permissão. <span><i class='fas fa-times'></i></span></p>"
  ], 500);
}

// ✅ Detecta MIME com finfo (melhor)
$tmp = $_FILES['logo']['tmp_name'];

$mime = '';
if (function_exists('finfo_open')) {
  $finfo = finfo_open(FILEINFO_MIME_TYPE);
  if ($finfo) {
    $mime = finfo_file($finfo, $tmp) ?: '';
    finfo_close($finfo);
  }
}

// fallback por extensão se mime vier vazio
$extFallback = strtolower(pathinfo($_FILES['logo']['name'] ?? '', PATHINFO_EXTENSION));
$allowed = [
  'image/png'  => 'png',
  'image/jpeg' => 'jpg',
  'image/webp' => 'webp'
];

if (!isset($allowed[$mime])) {
  // tenta decidir por extensão
  $mapExt = ['png' => 'image/png', 'jpg' => 'image/jpeg', 'jpeg' => 'image/jpeg', 'webp' => 'image/webp'];
  $mime2 = $mapExt[$extFallback] ?? '';
  if ($mime2 && isset($allowed[$mime2])) {
    $mime = $mime2;
  }
}

if (!isset($allowed[$mime])) {
  json_exit([
    "status"  => "alertanao",
    "message" => "<p class='alertanao'>Apenas PNG, JPEG e WEBP são permitidos! <span><i class='fas fa-times'></i></span></p>"
  ], 400);
}

$ext = $allowed[$mime];
$newFileName = uniqid('logo_', true) . '.' . $ext;
$destination = $uploadDir . DIRECTORY_SEPARATOR . $newFileName;

try {
  $pdo->beginTransaction();

  // Busca logo atual
  $stmt = $pdo->prepare("SELECT bet_logo FROM bet_adm_config WHERE id = 1 LIMIT 1");
  $stmt->execute();
  $logoAtual = $stmt->fetchColumn();

  // Move novo arquivo
  if (!move_uploaded_file($tmp, $destination)) {
    throw new Exception("Erro ao salvar a imagem (move_uploaded_file falhou).");
  }

  // Atualiza DB
  $stmt = $pdo->prepare("UPDATE bet_adm_config SET bet_logo = :logo WHERE id = 1");
  $stmt->bindValue(':logo', $newFileName, PDO::PARAM_STR);
  $stmt->execute();

  // Apaga antigo (depois de salvar o novo)
  if ($logoAtual) {
    $oldPath = $uploadDir . DIRECTORY_SEPARATOR . $logoAtual;
    if (is_file($oldPath)) @unlink($oldPath);
  }

  $pdo->commit();

  // Regenera CSRF
  $_SESSION['csrf_token_logo'] = bin2hex(random_bytes(32));

  json_exit([
    "status"   => "alertasim",
    "message"  => "<p class='alertasim'>Logo enviada com sucesso! <span><i class='fas fa-check'></i></span></p>",
    "new_csrf" => $_SESSION['csrf_token_logo'],
    "file"     => $newFileName
  ]);

} catch (Exception $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();

  // Se moveu arquivo e falhou depois, apaga
  if (isset($destination) && is_file($destination)) @unlink($destination);

  json_exit([
    "status"  => "alertanao",
    "message" => "<p class='alertanao'>Erro: ".htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8')." <span><i class='fas fa-times'></i></span></p>"
  ], 500);
}
