<?php
header('Content-Type: application/json; charset=utf-8');

function json_exit($arr, $code = 200) {
  http_response_code($code);
  echo json_encode($arr, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

// Só aceita POST AJAX
if (
  ($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST' ||
  empty($_SERVER['HTTP_X_REQUESTED_WITH']) ||
  strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) !== 'xmlhttprequest'
) {
  json_exit(["status" => "alertanao", "message" => "Acesso inválido"], 403);
}

session_name('adm_session');
session_start();

require_once '../../../includes/db.php';
require_once 'auth_ajax_adm.php';

// CSRF
function valida_token_csrf($form) {
  $token = $_POST['csrf_token'] ?? '';
  $sess  = $_SESSION["csrf_token_$form"] ?? '';
  return ($token && $sess && hash_equals($sess, $token));
}

$errors = [];

// Valida CSRF
if (!valida_token_csrf('favicon')) {
  $errors[] = "Falha de segurança. Recarregue a página.";
}

// Valida upload
if (!isset($_FILES['favicon']) || ($_FILES['favicon']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
  $errors[] = "Nenhum arquivo enviado.";
}

if (!empty($errors)) {
  json_exit([
    "status"  => "alertanao",
    "message" => "<p class='alertanao'>".implode("<br>", $errors)." <span><i class='fas fa-times'></i></span></p>"
  ], 400);
}

// Pasta segura
$uploadDir = realpath(__DIR__ . '/../../../imagens');
if ($uploadDir === false || !is_writable($uploadDir)) {
  json_exit([
    "status" => "alertanao",
    "message" => "<p class='alertanao'>Pasta de imagens inválida. <span><i class='fas fa-times'></i></span></p>"
  ], 500);
}

// Detecta MIME real
$tmp = $_FILES['favicon']['tmp_name'];
$finfo = finfo_open(FILEINFO_MIME_TYPE);
$mime  = finfo_file($finfo, $tmp);
finfo_close($finfo);

// Apenas PNG
if ($mime !== 'image/png') {
  json_exit([
    "status" => "alertanao",
    "message" => "<p class='alertanao'>Favicon deve ser PNG. <span><i class='fas fa-times'></i></span></p>"
  ], 400);
}

// Verifica tamanho máximo 48x48
$img = getimagesize($tmp);
if (!$img || $img[0] > 48 || $img[1] > 48) {
  json_exit([
    "status" => "alertanao",
    "message" => "<p class='alertanao'>Máximo permitido: 48x48 px. <span><i class='fas fa-times'></i></span></p>"
  ], 400);
}

$newFile = uniqid('favicon_', true) . '.png';
$dest = $uploadDir . DIRECTORY_SEPARATOR . $newFile;

try {
  $pdo->beginTransaction();

  // favicon atual
  $stmt = $pdo->prepare("SELECT bet_favicon FROM bet_adm_config WHERE id = 1 LIMIT 1");
  $stmt->execute();
  $old = $stmt->fetchColumn();

  // move novo
  if (!move_uploaded_file($tmp, $dest)) {
    throw new Exception("Falha ao salvar favicon.");
  }

  // atualiza DB
  $stmt = $pdo->prepare("UPDATE bet_adm_config SET bet_favicon = :f WHERE id = 1");
  $stmt->bindValue(':f', $newFile);
  $stmt->execute();

  // remove antigo
  if ($old) {
    $oldPath = $uploadDir . DIRECTORY_SEPARATOR . $old;
    if (is_file($oldPath)) @unlink($oldPath);
  }

  $pdo->commit();

  // novo CSRF
  $_SESSION['csrf_token_favicon'] = bin2hex(random_bytes(32));

  json_exit([
    "status"   => "alertasim",
    "message"  => "<p class='alertasim'>Favicon atualizado com sucesso! <span><i class='fas fa-check'></i></span></p>",
    "file"     => $newFile,
    "new_csrf" => $_SESSION['csrf_token_favicon']
  ]);

} catch (Exception $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  if (is_file($dest)) @unlink($dest);

  json_exit([
    "status" => "alertanao",
    "message" => "<p class='alertanao'>Erro: ".htmlspecialchars($e->getMessage())." <span><i class='fas fa-times'></i></span></p>"
  ], 500);
}
