<?php
// ============ DEBUG (use: /arquivo.php?debug=1) ============
$DEBUG = isset($_GET['debug']) && $_GET['debug'] == '1';
if ($DEBUG) {
    ini_set('display_errors', 1);
    error_reporting(E_ALL);
} else {
    ini_set('display_errors', 0);
    error_reporting(0);
}

// Impede acesso direto via navegador (GET)
if (
    $_SERVER['REQUEST_METHOD'] !== 'POST' ||
    empty($_SERVER['HTTP_X_REQUESTED_WITH']) ||
    strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) !== 'xmlhttprequest'
) {
    header("Location: /");
    exit;
}

session_start();

require_once '../../includes/db.php';
require_once '../../includes/config.php';

// =========================
// Helpers
// =========================
function json_fail($msg, $debugExtra = null) {
    $out = [
        "status"  => "alertanao",
        "message" => "<p class='alertanao'>{$msg} <span><i class='fas fa-times'></i></span></p>"
    ];
    if ($debugExtra) {
        // Nunca exponha em produção; só use com debug local
        $out["debug"] = $debugExtra;
    }
    header('Content-Type: application/json');
    echo json_encode($out);
    exit;
}

function valida_token_csrf($form_name) {
    $token = $_POST['csrf_token'] ?? '';
    return isset($_SESSION["csrf_token_$form_name"]) && hash_equals($_SESSION["csrf_token_$form_name"], $token);
}

function generateQRCodeBase64($data) {
    $url = 'https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=' . urlencode($data);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_FOLLOWLOCATION => true,
    ]);
    $imageData = curl_exec($ch);
    $httpCode  = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err       = curl_error($ch);
    curl_close($ch);

    if ($httpCode === 200 && $imageData !== false) {
        return 'data:image/png;base64,' . base64_encode($imageData);
    }

    // fallback
    $fallbackUrl = 'https://chart.googleapis.com/chart?chs=200x200&cht=qr&chl=' . urlencode($data);
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $fallbackUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
    ]);
    $imageData = curl_exec($ch);
    $httpCode  = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err2      = curl_error($ch);
    curl_close($ch);

    if ($httpCode === 200 && $imageData !== false) {
        return 'data:image/png;base64,' . base64_encode($imageData);
    }

    // último fallback
    return 'data:image/svg+xml;base64,' . base64_encode(
        '<svg width="200" height="200" xmlns="http://www.w3.org/2000/svg">
            <rect width="200" height="200" fill="#f0f0f0"/>
            <text x="100" y="100" text-anchor="middle" dominant-baseline="central"
                  font-family="Arial" font-size="14">QR Code Indisponível</text>
         </svg>'
    );
}

// =========================
// Validação
// =========================
if (!valida_token_csrf('deposito')) {
    json_fail("Falha. Por favor, tente novamente.");
}

$depositoRaw = trim($_POST["deposito"] ?? '');
if ($depositoRaw === '') {
    json_fail("O campo depósito é obrigatório!");
}

// limpa moeda
$cleanValue = str_replace(['R$', ' ', ' '], ['', '', ''], $depositoRaw);
$cleanValue = str_replace('.', '', $cleanValue);
$cleanValue = str_replace(',', '.', $cleanValue);

if (!preg_match('/^\d+(\.\d{1,2})?$/', $cleanValue)) {
    json_fail("Valor não aceito!");
}

$valorDeposito = (float)$cleanValue;
if ($valorDeposito < (float)$ValorDeposito) {
    json_fail("O depósito mínimo é de R$ " . number_format($ValorDeposito, 2, ',', '.') . " reais!");
}
if ($valorDeposito > 10000) {
    json_fail("O depósito máximo é de R$ 10.000,00 reais!");
}

$usuario_id = $_SESSION['usuario_id'] ?? null;
if (!$usuario_id) {
    json_fail("Sessão expirada. Faça login novamente.");
}

// =========================
// ClashPay
// =========================
try {
    $pdo->beginTransaction();

    $stmt = $pdo->prepare("SELECT bet_nome, bet_email, bet_cpf, bet_origem FROM bet_usuarios WHERE id = ? LIMIT 1");
    $stmt->execute([$usuario_id]);
    $cliente = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$cliente) {
        throw new Exception("Cliente não encontrado no banco.");
    }

    if (empty($ClientIdClashPay) || empty($ClientSecretClashPay)) {
        throw new Exception("Credenciais ClashPay não configuradas.");
    }

    // 1) AUTH
    $credentials = base64_encode($ClientIdClashPay . ':' . $ClientSecretClashPay);

    $authCh = curl_init('https://api.royalbenk.com/v2/oauth/token');
    curl_setopt_array($authCh, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => [
            'Authorization: Basic ' . $credentials,
            'Content-Type: application/json'
        ],
        // ⚠️ Se a API exigir body, coloque aqui.
        // Exemplo comum:
        // CURLOPT_POSTFIELDS => json_encode(['grant_type' => 'client_credentials']),
        CURLOPT_TIMEOUT => 20,
    ]);

    $authResponse = curl_exec($authCh);
    $authErr      = curl_error($authCh);
    $authHttp     = curl_getinfo($authCh, CURLINFO_HTTP_CODE);
    curl_close($authCh);

    if ($authResponse === false) {
        throw new Exception("Falha cURL AUTH: " . $authErr);
    }

    $authResult = json_decode($authResponse, true);
    $jwtToken   = $authResult['access_token'] ?? null;

    if (!$jwtToken) {
        throw new Exception("Erro na autenticação JWT. HTTP={$authHttp} Resposta=" . $authResponse);
    }

    // 2) GERAR PIX
    $external_id  = uniqid('dep_' . $usuario_id . '_', true);
    $callback_url = 'https://' . $_SERVER['HTTP_HOST'] . '/dashboard/php/callback_deposito.php';

    $payload = json_encode([
        'amount'       => (float)$valorDeposito,
        'external_id'  => $external_id,
        'payerQuestion'=> 'Depósito - ' . ($NomeSite ?? 'Site'),
        'postbackUrl'  => $callback_url,
        'payer' => [
            'name'     => $cliente['bet_nome'],
            'document' => (int)preg_replace('/\D/', '', $cliente['bet_cpf']),
            'email'    => $cliente['bet_email']
        ]
    ]);

    $ch = curl_init('https://api.royalbenk.com/v2/pix/qrcode');
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'Authorization: Bearer ' . $jwtToken,
            'Content-Type: application/json'
        ],
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $payload,
        CURLOPT_TIMEOUT => 20,
    ]);

    $pixResponse = curl_exec($ch);
    $pixErr      = curl_error($ch);
    $pixHttp     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($pixResponse === false) {
        throw new Exception("Falha cURL PIX: " . $pixErr);
    }

    $resultadoPix = json_decode($pixResponse, true);
    if (!is_array($resultadoPix)) {
        throw new Exception("Resposta PIX não é JSON. HTTP={$pixHttp} Body=" . $pixResponse);
    }

    $id_transacao = $resultadoPix['transactionId'] ?? null;
    $qr_code      = $resultadoPix['qrcode'] ?? null;

    if (!$id_transacao || !$qr_code) {
        $msgApi = $resultadoPix['message'] ?? $resultadoPix['error'] ?? 'Erro desconhecido';
        throw new Exception("Erro na geração do QR Code. HTTP={$pixHttp} Msg={$msgApi} Body=" . $pixResponse);
    }

    $qr_code_base64 = generateQRCodeBase64($qr_code);

    // 3) SALVAR TRANSACAO
    $origem    = $cliente['bet_origem'] ?? '';
    $ipUsuario = $_SERVER['REMOTE_ADDR'] ?? '';
    $fbc       = $_COOKIE['_fbc'] ?? '';
    $fbp       = $_COOKIE['_fbp'] ?? '';

    $insert = $pdo->prepare("
        INSERT INTO bet_transacoes 
        (bet_usuario, bet_id_transacao, bet_valor, bet_tipo, bet_status, bet_data, bet_origem, bet_ip, bet_fbc, bet_fbp)
        VALUES
        (:usuario, :id_transacao, :valor, :tipo, 'Pendente', NOW(), :origem, :ip, :fbc, :fbp)
    ");
    $insert->execute([
        ':usuario'      => $usuario_id,
        ':id_transacao' => $id_transacao,
        ':valor'        => $valorDeposito,
        ':tipo'         => 'Deposito',
        ':origem'       => $origem,
        ':ip'           => $ipUsuario,
        ':fbc'          => $fbc,
        ':fbp'          => $fbp
    ]);

    $pdo->commit();

    // 4) RETORNAR HTML DO MODAL
    $corPrincipalSafe = isset($corPrincipal) ? $corPrincipal : '#00C774';

    $html = '
    <div class="modal-content">
        <span class="close-modal"><i class="fas fa-times"></i></span>
        <h2>Depósito</h2>
        <div class="pix-container">
            <p>Escaneie o QR Code abaixo ou copie o código PIX no aplicativo do seu banco:</p>

            <div style="text-align: center; margin: 20px 0;">
                <img src="' . $qr_code_base64 . '" alt="QR Code PIX" style="max-width:200px;height:auto;border:1px solid #ddd;border-radius:8px;" />
            </div>

            <p style="text-align: center; margin-bottom: 15px;">Ou copie o código PIX abaixo:</p>

            <div class="form-row">
                <div class="input-icon">
                    <i class="fas fa-clipboard-check"></i>
                    <input type="text" id="pixLink" value="' . htmlspecialchars($qr_code, ENT_QUOTES, 'UTF-8') . '" readonly>
                </div>
            </div>

            <input type="hidden" name="txidmodal" value="' . htmlspecialchars($id_transacao, ENT_QUOTES, 'UTF-8') . '">

            <button class="submit-button BotaoCopiaPix" onclick="copiarCodigo()">Copiar Código PIX</button>

            <p style="text-align:center;">
                Valor do PIX: R$ <strong style="color:' . $corPrincipalSafe . '">' . number_format($valorDeposito, 2, ',', '.') . '</strong> reais.
            </p>

            <p style="text-align:center;font-size:14px;margin-top:10px;">
                <strong>ID da Transação:</strong> ' . htmlspecialchars($external_id, ENT_QUOTES, 'UTF-8') . '<br>
                <strong>Status:</strong> Pendente
            </p>
        </div>
    </div>';

    $html .= "
    <script type=\"text/javascript\">
      (function(){
        function checkStatus(interval) {
          const txidEl = document.querySelector('input[name=\"txidmodal\"]');
          if (!txidEl) return;

          const txid = txidEl.value;

          $.ajax({
            url: 'php/check_pagamento.php?txid=' + encodeURIComponent(txid) + '&t=' + Date.now(),
            type: 'GET',
            dataType: 'json',
            success: function(response) {
              if (response && response.status === 'Aprovado') {
                clearInterval(interval);
                window.location.href = '/dashboard/';
              }
            }
          });
        }

        const interval = setInterval(function(){ checkStatus(interval); }, 3000);
      })();
    </script>
    ";

    $_SESSION['csrf_token_deposito'] = bin2hex(random_bytes(32));

    header('Content-Type: application/json');
    echo json_encode([
        "status" => "alertasim",
        "html"   => $html
    ]);
    exit;

} catch (Exception $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();

    error_log("[DEPOSITO CLASHPAY ERRO] " . $e->getMessage());

    json_fail(
        "Erro ao processar depósito. Tente novamente.",
        // debug extra só se você estiver usando ?debug=1
        (isset($DEBUG) && $DEBUG) ? $e->getMessage() : null
    );
}
