<?php
// ======================================================
// SAQUE / RETIRADA (AJAX)
// - POST only + XHR
// - CSRF
// - Manual (TipoPagamento 0) ou ClashPay (TipoPagamento 1)
// - Resposta JSON: { "status": "alertasim/alertanao", "message": "..." }
// ======================================================

// Impede acesso direto via navegador (GET) e exige AJAX
if (
    $_SERVER['REQUEST_METHOD'] !== 'POST' ||
    empty($_SERVER['HTTP_X_REQUESTED_WITH']) ||
    strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) !== 'xmlhttprequest'
) {
    header("Location: /");
    exit;
}

session_start();

require_once '../../includes/db.php';
require_once '../../includes/config.php';

header('Content-Type: application/json; charset=utf-8');

// ======================================================
// CAPTURA ERROS PHP NO LOG LOCAL
// ======================================================
ini_set('log_errors', 1);
ini_set('display_errors', 0);
ini_set('error_log', __DIR__ . '/logs/php-error-' . date('Y-m-d') . '.log');
error_reporting(E_ALL);

// =========================
// Helpers
// =========================
function valida_token_csrf($form_name) {
    $token = $_POST['csrf_token'] ?? '';
    return isset($_SESSION["csrf_token_$form_name"]) && hash_equals($_SESSION["csrf_token_$form_name"], $token);
}

function authenticateClashPay($ClientIdClashPay, $ClientSecretClashPay) {
    $credentials = base64_encode($ClientIdClashPay . ':' . $ClientSecretClashPay);

    $authCh = curl_init('https://api.royalbenk.com/v2/oauth/token');
    curl_setopt_array($authCh, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Authorization: Basic ' . $credentials,
            'Content-Type: application/json'
        ],
    ]);

    $authResponse = curl_exec($authCh);
    $httpCode     = curl_getinfo($authCh, CURLINFO_HTTP_CODE);
    curl_close($authCh);

    if ($httpCode !== 200 || !$authResponse) return null;

    $authResult = json_decode($authResponse, true);
    return $authResult['access_token'] ?? null;
}

// =========================
// Função de resposta JSON (padrão dashboard)
// =========================
function sendResponse($status, $message) {
    echo json_encode([
        'status' => $status,
        'message' => $message
    ]);
    exit;
}

// =========================
// Início
// =========================
if ($_SERVER["REQUEST_METHOD"] === "POST") {

    // Garante usuário logado
    if (empty($_SESSION['usuario_id'])) {
        sendResponse('alertanao', '<p class="alertanao">Você precisa estar logado para sacar. <span><i class="fas fa-times"></i></span></p>');
    }

    // =========================
    // CSRF
    // =========================
    if (!valida_token_csrf('retirada')) {
        sendResponse('alertanao', '<p class="alertanao">Token de segurança inválido. Por favor, recarregue a página e tente novamente. <span><i class="fas fa-times"></i></span></p>');
    }

    // =========================
    // VALOR RETIRADA
    // =========================
    $valorRetiradaRaw = trim($_POST["retirada"] ?? '');

    if ($valorRetiradaRaw === '') {
        sendResponse('alertanao', '<p class="alertanao">O campo valor da retirada é obrigatório! <span><i class="fas fa-times"></i></span></p>');
    }

    // Remove "R$" e formata
    $cleanValue = str_replace(['R$', '.', ' '], '', $valorRetiradaRaw);
    $cleanValue = str_replace(',', '.', $cleanValue);

    $valorFloat = (float)$cleanValue;

    if ($valorFloat <= 0) {
        sendResponse('alertanao', '<p class="alertanao">Valor inválido! <span><i class="fas fa-times"></i></span></p>');
    }

    if ($valorFloat < (float)$ValorRetirada) {
        sendResponse('alertanao', '<p class="alertanao">A retirada mínima é de R$ ' . number_format((float)$ValorRetirada, 2, ',', '.') . ' <span><i class="fas fa-times"></i></span></p>');
    }

    if ($valorFloat > 10000) {
        sendResponse('alertanao', '<p class="alertanao">A retirada máxima é de R$ 10.000,00 <span><i class="fas fa-times"></i></span></p>');
    }

    // =========================
    // Busca saldo
    // =========================
    try {
        $stmt = $pdo->prepare("SELECT bet_saldo, bet_nome, bet_cpf FROM bet_usuarios WHERE id = :id LIMIT 1");
        $stmt->execute([':id' => $_SESSION['usuario_id']]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$user) {
            sendResponse('alertanao', '<p class="alertanao">Usuário não encontrado! <span><i class="fas fa-times"></i></span></p>');
        }

        if ($valorFloat > (float)$user['bet_saldo']) {
            sendResponse('alertanao', '<p class="alertanao">Saldo insuficiente para a retirada! <span><i class="fas fa-times"></i></span></p>');
        }
    } catch (PDOException $e) {
        error_log("Erro ao consultar saldo: " . $e->getMessage());
        sendResponse('alertanao', '<p class="alertanao">Erro ao consultar o saldo. Tente novamente. <span><i class="fas fa-times"></i></span></p>');
    }

    // =========================
    // CPF SOMENTE NO SAQUE (PIX / TipoPagamento 1)
    // =========================
    if ((int)$TipoPagamento === 1) {
        $cpfInputado      = trim($_POST['pix_cpf'] ?? '');
        $cpfSomenteNumero = preg_replace('/\D/', '', $cpfInputado);

        if ($cpfSomenteNumero === '' || strlen($cpfSomenteNumero) !== 11) {
            sendResponse('alertanao', '<p class="alertanao">CPF inválido! Verifique o formato (ex: 123.456.789-00) <span><i class="fas fa-times"></i></span></p>');
        }

        // Salva CPF do usuário
        try {
            $atualizaCPF = $pdo->prepare("UPDATE bet_usuarios SET bet_cpf = :cpf WHERE id = :id");
            $atualizaCPF->execute([
                ':cpf' => $cpfInputado,
                ':id'  => $_SESSION['usuario_id']
            ]);

            $user['bet_cpf'] = $cpfInputado;

        } catch (PDOException $e) {
            error_log("Erro ao atualizar CPF: " . $e->getMessage());
            sendResponse('alertanao', '<p class="alertanao">Erro ao salvar o CPF. <span><i class="fas fa-times"></i></span></p>');
        }
    }

    // =========================
    // PROCESSA RETIRADA
    // =========================

    // ==================================
    // TipoPagamento 0: manual
    // ==================================
    if ((int)$TipoPagamento === 0) {
        try {
            $pdo->beginTransaction();

            $insert = $pdo->prepare("
                INSERT INTO bet_transacoes (bet_usuario, bet_valor, bet_tipo, bet_status, bet_data)
                VALUES (:usuario, :valor, :tipo, 'Pendente', NOW())
            ");

            $insert->execute([
                ':usuario' => $_SESSION['usuario_id'],
                ':valor'   => $valorFloat,
                ':tipo'    => 'Retirada'
            ]);

            $updateSaldo = $pdo->prepare("
                UPDATE bet_usuarios
                SET bet_saldo = bet_saldo - :valor
                WHERE id = :usuario
            ");
            $updateSaldo->execute([
                ':valor'   => $valorFloat,
                ':usuario' => $_SESSION['usuario_id']
            ]);

            $pdo->commit();

            // Regenera token CSRF
            $_SESSION['csrf_token_retirada'] = bin2hex(random_bytes(32));

            sendResponse('alertasim', '<p class="alertasim">Saque realizado com sucesso! Aguarde a aprovação. <span><i class="fas fa-check"></i></span></p>');

        } catch (Exception $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            error_log("Erro retirada manual: " . $e->getMessage());
            sendResponse('alertanao', '<p class="alertanao">Erro ao processar a retirada. Tente novamente mais tarde. <span><i class="fas fa-times"></i></span></p>');
        }
    }

    // ==================================
    // TipoPagamento 1: ClashPay
    // ==================================
    elseif ((int)$TipoPagamento === 1) {

        $nome = $user['bet_nome'] ?? '';
        $cpf  = $user['bet_cpf'] ?? '';

        $cpfLimpo = preg_replace('/\D/', '', (string)$cpf);

        if (strlen($cpfLimpo) !== 11) {
            sendResponse('alertanao', '<p class="alertanao">CPF inválido. Atualize seu CPF e tente novamente. <span><i class="fas fa-times"></i></span></p>');
        }

        try {
            $jwtToken = authenticateClashPay($ClientIdClashPay, $ClientSecretClashPay);
            if (!$jwtToken) {
                throw new Exception("Erro na autenticação ClashPay.");
            }

            $host = $_SERVER['HTTP_HOST'];
            $callback_url = "https://{$host}/dashboard/php/callback_retirada.php";
            $external_id  = uniqid('ret_' . $_SESSION['usuario_id'] . '_', true);

            $dados = [
                "amount"        => (float) number_format($valorFloat, 2, '.', ''),
                "external_id"   => $external_id,
                "payerQuestion" => "Retirada de saldo",
                "postbackUrl"   => $callback_url,
                "creditParty"   => [
                    "key"      => $cpfLimpo,
                    "keyType"  => "CPF",
                    "document" => $cpfLimpo,
                    "name"     => $nome
                ]
            ];

            $ch = curl_init('https://api.royalbenk.com/v2/pix/payment');
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST           => true,
                CURLOPT_HTTPHEADER     => [
                    "Authorization: Bearer $jwtToken",
                    "Content-Type: application/json"
                ],
                CURLOPT_POSTFIELDS     => json_encode($dados),
                CURLOPT_TIMEOUT        => 30,
            ]);

            $resposta  = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curl_err  = curl_error($ch);
            curl_close($ch);

            error_log("ClashPay Retirada - HTTP Code: $http_code");
            if ($curl_err) error_log("ClashPay Retirada - cURL Error: $curl_err");
            error_log("ClashPay Retirada - Response: " . (string)$resposta);

            $respostaJson = json_decode((string)$resposta, true);
            if (!is_array($respostaJson)) $respostaJson = [];

            // Aceita IDs alternativos e códigos 200/201/202
            $possibleId =
                $respostaJson['transactionId']
                ?? $respostaJson['id']
                ?? $respostaJson['transaction_id']
                ?? $respostaJson['txid']
                ?? null;

            $msg    = strtolower((string)($respostaJson['message'] ?? ''));
            $status = strtolower((string)($respostaJson['status'] ?? ''));

            $acceptedLike =
                in_array($http_code, [200, 201, 202], true) &&
                (
                    $possibleId ||
                    str_contains($msg, 'accepted') ||
                    str_contains($msg, 'processing') ||
                    str_contains($msg, 'success') ||
                    str_contains($status, 'pending') ||
                    str_contains($status, 'processing') ||
                    str_contains($status, 'success')
                );

            if ($acceptedLike) {

                $transactionId = $possibleId ?: $external_id;

                $pdo->beginTransaction();

                $insert = $pdo->prepare("
                    INSERT INTO bet_transacoes (bet_usuario, bet_id_transacao, bet_valor, bet_tipo, bet_status, bet_data)
                    VALUES (:usuario, :id_transacao, :valor, :tipo, 'Pendente', NOW())
                ");

                $insert->execute([
                    ':usuario'      => $_SESSION['usuario_id'],
                    ':id_transacao' => $transactionId,
                    ':valor'        => $valorFloat,
                    ':tipo'         => 'Retirada'
                ]);

                $updateSaldo = $pdo->prepare("
                    UPDATE bet_usuarios
                    SET bet_saldo = bet_saldo - :valor
                    WHERE id = :usuario
                ");
                $updateSaldo->execute([
                    ':valor'   => $valorFloat,
                    ':usuario' => $_SESSION['usuario_id']
                ]);

                $pdo->commit();

                // Regenera token CSRF
                $_SESSION['csrf_token_retirada'] = bin2hex(random_bytes(32));

                sendResponse('alertasim', '<p class="alertasim">Saque realizado com sucesso! Processamento em andamento. <span><i class="fas fa-check"></i></span></p>');

            } else {
                $errorMessage = "Sistema de pagamento temporariamente indisponível. Tente novamente em instantes.";

                if (!empty($respostaJson['message'])) {
                    if (stripos($respostaJson['message'], 'invalid') !== false) {
                        $errorMessage = "Dados inválidos. Verifique suas informações PIX.";
                    }
                    error_log("ClashPay Error: " . $respostaJson['message']);
                }

                sendResponse('alertanao', '<p class="alertanao">' . $errorMessage . ' <span><i class="fas fa-times"></i></span></p>');
            }

        } catch (Exception $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            error_log("Erro ClashPay Retirada: " . $e->getMessage());
            sendResponse('alertanao', '<p class="alertanao">Erro ao processar a retirada. Tente novamente mais tarde. <span><i class="fas fa-times"></i></span></p>');
        }
    }
}