<?php
/**
 * dashboard/php/jbicho_minhas_apostas.php
 * Retorna apostas do usuário via JSON.
 * Parâmetros GET:
 *   - status: pendente | ganhou | perdeu | todos (default: todos)
 *   - pagina: número da página (default: 1)
 *   - limite: itens por página (default: 10)
 */
session_start();
header('Content-Type: application/json');

$base = __DIR__ . '/../../includes/';
require_once $base . 'db.php';
require_once $base . 'config.php';

if (!isset($_SESSION['usuario_id'])) {
    echo json_encode(['success' => false, 'message' => 'Não autenticado.']);
    exit;
}

$uid    = (int)$_SESSION['usuario_id'];
$status = $_GET['status'] ?? 'todos';
$pagina = max(1, (int)($_GET['pagina'] ?? 1));
$limite = min(20, max(5, (int)($_GET['limite'] ?? 10)));
$offset = ($pagina - 1) * $limite;

// Whitelist status
$status_validos = ['pendente', 'ganhou', 'perdeu', 'todos'];
if (!in_array($status, $status_validos)) $status = 'todos';

try {
    // Monta WHERE
    $where = "a.usuario_id = ?";
    $params = [$uid];
    if ($status !== 'todos') {
        $where .= " AND a.status = ?";
        $params[] = $status;
    }

    // Total de registros (para saber se tem mais página)
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM jbicho_apostas a WHERE $where");
    $stmt->execute($params);
    $total = (int)$stmt->fetchColumn();

    // Busca apostas com JOIN nas tabelas de modalidade e loteria
    // MariaDB não aceita placeholders ? em LIMIT/OFFSET — usa inteiros diretamente (já validados acima)
    $stmt = $pdo->prepare("
        SELECT
            a.id,
            a.modalidade_id,
            m.nome AS modalidade_nome,
            a.loteria_id,
            l.sigla AS loteria_sigla,
            l.nome  AS loteria_nome,
            a.data_sorteio,
            a.colocacoes,
            a.palpites,
            a.valor,
            a.cotacao_usada,
            a.status,
            a.premiacao,
            a.criado_em
        FROM jbicho_apostas a
        INNER JOIN jbicho_modalidades m ON a.modalidade_id = m.id
        INNER JOIN jbicho_loterias l    ON a.loteria_id    = l.id
        WHERE $where
        ORDER BY a.criado_em DESC
        LIMIT {$limite} OFFSET {$offset}
    ");
    $stmt->execute($params);
    $apostas = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Decodifica palpites e colocações para cada aposta
    foreach ($apostas as &$a) {
        $a['palpites'] = json_decode($a['palpites'], true) ?? [];

        // Decodifica bitmask de colocações
        $cols = [];
        for ($i = 0; $i < 7; $i++) {
            if ($a['colocacoes'] & (1 << $i)) {
                $cols[] = ($i + 1);
            }
        }
        $a['colocacoes_array'] = $cols;

        // Formata valores
        $a['valor_fmt']     = number_format((float)$a['valor'], 2, ',', '.');
        $a['premiacao_fmt'] = number_format((float)$a['premiacao'], 2, ',', '.');
        $a['cotacao_fmt']   = number_format((float)$a['cotacao_usada'], 2, ',', '.');

        // Formata data
        $a['data_fmt'] = date('d/m/Y', strtotime($a['data_sorteio']));
    }
    unset($a);

    $tem_mais = ($offset + $limite) < $total;

    echo json_encode([
        'success'  => true,
        'apostas'  => $apostas,
        'total'    => $total,
        'pagina'   => $pagina,
        'tem_mais' => $tem_mais
    ]);

} catch (Throwable $e) {
    echo json_encode(['success' => false, 'message' => 'Erro: ' . $e->getMessage()]);
}