<?php
/**
 * dashboard/php/jbicho_apostar.php
 * Recebe aposta via POST (AJAX), valida, desconta saldo, persiste.
 */
session_start();
header('Content-Type: application/json');

// Caminho: este arquivo está em dashboard/php/
// includes/ está uma pasta acima de dashboard/ → ../../includes/
$base = __DIR__ . '/../../includes/';

if (!file_exists($base . 'db.php')) {
    echo json_encode(['success' => false, 'message' => 'Erro: db.php não encontrado.']);
    exit;
}

require_once $base . 'db.php';
require_once $base . 'config.php';

// ── Autenticação ──────────────────────────────────────────────
if (!isset($_SESSION['usuario_id'])) {
    echo json_encode(['success' => false, 'message' => 'Sessão expirada. Faça login novamente.']);
    exit;
}
$uid = (int)$_SESSION['usuario_id'];

// ── CSRF ──────────────────────────────────────────────────────
if (($_POST['csrf_token'] ?? '') !== ($_SESSION['csrf_token_jbicho'] ?? '')) {
    echo json_encode(['success' => false, 'message' => 'Token inválido. Recarregue a página.']);
    exit;
}
// Regera token após uso
$_SESSION['csrf_token_jbicho'] = bin2hex(random_bytes(32));

// ── Dados da requisição ───────────────────────────────────────
$modalidade_id = (int)($_POST['modalidade_id'] ?? 0);
$loteria_id    = (int)($_POST['loteria_id']    ?? 0);
$data_sorteio  = $_POST['data_sorteio']        ?? '';
$colocacoes    = (int)($_POST['colocacoes']    ?? 0);
$palpites_json = $_POST['palpites']            ?? '[]';
$valor         = (float)($_POST['valor']       ?? 0);

// ── Validações básicas ───────────────────────────────────────
if ($modalidade_id === 0 || $loteria_id === 0 || $valor < 0.10) {
    echo json_encode(['success' => false, 'message' => 'Dados incompletos.']);
    exit;
}
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $data_sorteio)) {
    echo json_encode(['success' => false, 'message' => 'Data inválida.']);
    exit;
}
if ($colocacoes < 1 || $colocacoes > 127) {
    echo json_encode(['success' => false, 'message' => 'Colocações inválidas.']);
    exit;
}

$palpites = json_decode($palpites_json, true);
if (!is_array($palpites) || empty($palpites)) {
    echo json_encode(['success' => false, 'message' => 'Palpites inválidos.']);
    exit;
}

// ── Busca modalidade & cotação ────────────────────────────────
try {
    $stmt = $pdo->prepare("SELECT * FROM jbicho_modalidades WHERE id = ? AND ativo = 1");
    $stmt->execute([$modalidade_id]);
    $modal = $stmt->fetch(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    echo json_encode(['success' => false, 'message' => 'Erro interno: ' . $e->getMessage()]);
    exit;
}
if (!$modal) {
    echo json_encode(['success' => false, 'message' => 'Modalidade não encontrada.']);
    exit;
}

// Valida quantidade de palpites
if ((int)count($palpites) !== (int)$modal['qtd_palpites']) {
    echo json_encode(['success' => false, 'message' => 'Número de palpites incorreto para esta modalidade.']);
    exit;
}

// Valida loteria
try {
    $stmt = $pdo->prepare("SELECT id FROM jbicho_loterias WHERE id = ? AND ativo = 1");
    $stmt->execute([$loteria_id]);
    if (!$stmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Loteria não encontrada.']);
        exit;
    }
} catch (Throwable $e) {
    echo json_encode(['success' => false, 'message' => 'Erro interno: ' . $e->getMessage()]);
    exit;
}

// Valor mínimo
$valor_minimo = 1.00;
if ($valor < $valor_minimo) {
    echo json_encode(['success' => false, 'message' => "O valor mínimo de aposta é R$ " . number_format($valor_minimo, 2, ',', '.')]);
    exit;
}

// ── Saldo suficiente? ─────────────────────────────────────────
try {
    $pdo->beginTransaction();

    $stmt = $pdo->prepare("SELECT bet_saldo FROM bet_usuarios WHERE id = ? FOR UPDATE");
    $stmt->execute([$uid]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Usuário não encontrado.']);
        exit;
    }
    $saldo_atual = (float)$row['bet_saldo'];

    if ($saldo_atual < $valor) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Saldo insuficiente. Deposite mais para continuar.']);
        exit;
    }

    // Desconta saldo
    $stmt = $pdo->prepare("UPDATE bet_usuarios SET bet_saldo = bet_saldo - ? WHERE id = ?");
    $stmt->execute([$valor, $uid]);

    // Persiste aposta
    $stmt = $pdo->prepare("
        INSERT INTO jbicho_apostas
            (usuario_id, modalidade_id, loteria_id, data_sorteio, colocacoes, palpites, valor, cotacao_usada)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([
        $uid,
        $modalidade_id,
        $loteria_id,
        $data_sorteio,
        $colocacoes,
        json_encode($palpites),
        $valor,
        $modal['cotacao']
    ]);
    $aposta_id = $pdo->lastInsertId();

    $pdo->commit();

    // Busca nome da loteria para retorno
    $stmt = $pdo->prepare("SELECT nome, sigla FROM jbicho_loterias WHERE id = ?");
    $stmt->execute([$loteria_id]);
    $lot = $stmt->fetch(PDO::FETCH_ASSOC);

    echo json_encode([
        'success'    => true,
        'message'    => 'Aposta realizada com sucesso!',
        'aposta_id'  => $aposta_id,
        'novo_saldo' => number_format($saldo_atual - $valor, 2, ',', '.'),
        'modalidade' => $modal['nome'],
        'loteria'    => ($lot ? $lot['sigla'] . ' - ' . $lot['nome'] : ''),
        'loteria_id' => $loteria_id,
        'data'       => date('d/m/Y', strtotime($data_sorteio)),
        'valor'      => number_format($valor, 2, ',', '.'),
        'palpites'   => $palpites,
        'cotacao'    => number_format((float)$modal['cotacao'], 2, ',', '.')
    ]);

} catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    echo json_encode(['success' => false, 'message' => 'Erro ao processar aposta: ' . $e->getMessage()]);
}