<?php
/**
 * get_saldo.php
 *
 * Polling leve (a cada 10 s pelo frontend).
 * Retorna:
 *   - saldo atual do usuário
 *   - arrecadacao_atual da sala (para modo meta)
 *   - status da sala
 *   - lista de ganhadores recentes (para atualizar painel)
 *
 * GET  ?sala_id=<id>
 *
 * Resposta JSON:
 *   {
 *     status: "success",
 *     saldo: "1.234,56",
 *     saldo_float: 1234.56,
 *     arrecadacao_atual: 87.50,
 *     sala_status: "em_andamento",
 *     jogo_em_andamento: true,
 *     ganhadores: [...]
 *   }
 */

if (session_status() === PHP_SESSION_NONE) session_start();

// ─── AUTH ─────────────────────────────────────────────────────
if (!isset($_SESSION['usuario_id'])) {
    require_once '../../includes/db.php';
    if (isset($_COOKIE['auth_token'])) {
        $stmt = $pdo->prepare("SELECT id, bet_status FROM bet_usuarios WHERE bet_token = ?");
        $stmt->execute([$_COOKIE['auth_token']]);
        $u = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($u && $u['bet_status'] == 1) $_SESSION['usuario_id'] = $u['id'];
    }
    if (!isset($_SESSION['usuario_id'])) {
        http_response_code(401);
        exit(json_encode(['status' => 'erro']));
    }
}

require_once '../../includes/db.php';

$sala_id = (int)($_GET['sala_id'] ?? 0);

// ─── SALDO DO USUÁRIO ─────────────────────────────────────────
$stmt = $pdo->prepare("SELECT bet_saldo FROM bet_usuarios WHERE id = ?");
$stmt->execute([$_SESSION['usuario_id']]);
$saldo = (float)($stmt->fetchColumn() ?: 0);

// ─── DADOS DA SALA ────────────────────────────────────────────
$sala_status        = '';
$arrecadacao_atual  = 0.0;
$jogo_em_andamento  = false;

if ($sala_id > 0) {
    $stmt = $pdo->prepare("SELECT status, arrecadacao_atual FROM salas_bingo WHERE id = ?");
    $stmt->execute([$sala_id]);
    $sala = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($sala) {
        $sala_status       = $sala['status'];
        $arrecadacao_atual = (float)($sala['arrecadacao_atual'] ?? 0);
        $jogo_em_andamento = ($sala_status === 'em_andamento');
    }
}

// ─── GANHADORES RECENTES ──────────────────────────────────────
$ganhadores = [];
try {
    if ($sala_id > 0 && $jogo_em_andamento) {
        // Ganhadores apenas da sala atual (durante jogo)
        $stmt = $pdo->prepare("
            SELECT u.bet_nome AS nome_completo, t.valor, t.descricao, t.created_at
            FROM transacoes t
            JOIN bet_usuarios u ON t.usuario_id = u.id
            WHERE t.tipo = 'premio_bingo'
              AND t.valor > 0
              AND t.sala_id = ?
              AND (t.descricao LIKE '%Quadra%' OR t.descricao LIKE '%Quina%' OR t.descricao LIKE '%Cartela cheia%')
            ORDER BY t.created_at DESC
            LIMIT 15
        ");
        $stmt->execute([$sala_id]);
    } else {
        // Ganhadores globais recentes (quando não há jogo ao vivo)
        $stmt = $pdo->prepare("
            SELECT u.bet_nome AS nome_completo, t.valor, t.descricao, t.created_at
            FROM transacoes t
            JOIN bet_usuarios u ON t.usuario_id = u.id
            WHERE t.tipo = 'premio_bingo'
              AND t.valor > 0
            ORDER BY t.created_at DESC
            LIMIT 10
        ");
        $stmt->execute();
    }
    $ganhadores = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    // Silenciado — não falha o polling por causa dos ganhadores
}

// ─── RESPOSTA ─────────────────────────────────────────────────
exit(json_encode([
    'status'            => 'success',
    'saldo'             => number_format($saldo, 2, ',', '.'),
    'saldo_float'       => $saldo,
    'arrecadacao_atual' => $arrecadacao_atual,
    'sala_status'       => $sala_status,
    'jogo_em_andamento' => $jogo_em_andamento,
    'ganhadores'        => $ganhadores
]));
