<?php
/**
 * ajax/sorteio_server.php
 *
 * Endpoint do sorteio de bolas.
 *
 * POST JSON:
 *   { sala_id }                           → retorna próxima bola sorteada (se chegou a hora)
 *   { sala_id, bingo_feito:true, user_id }→ fecha sala como "bingo_fechado"
 *
 * Resposta JSON:
 *   { status, numero?, total_bolas, numeros_sorteados, wait_ms?, next_in_ms? }
 *
 * Status:
 *   ok             – bola retornada
 *   aguardando     – sala ainda não iniciou
 *   aguardando_bola– ainda não está na hora de soltar a próxima bola (controle servidor)
 *   finalizado     – 75 bolas sorteadas
 *   bingo_fechado  – sala fechada por bingo
 *   erro           – algo deu errado
 */

header('Content-Type: application/json; charset=utf-8');

if (session_status() === PHP_SESSION_NONE) session_start();

// ─── AUTH ─────────────────────────────────────────────────────
if (!isset($_SESSION['usuario_id'])) {
    require_once '../../../includes/db.php';

    if (isset($_COOKIE['auth_token'])) {
        $stmt = $pdo->prepare("SELECT id, bet_status FROM bet_usuarios WHERE bet_token = ?");
        $stmt->execute([$_COOKIE['auth_token']]);
        $u = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($u && (int)$u['bet_status'] === 1) {
            $_SESSION['usuario_id'] = (int)$u['id'];
        }
    }

    if (!isset($_SESSION['usuario_id'])) {
        http_response_code(401);
        echo json_encode(['status' => 'erro', 'message' => 'Não autenticado']);
        exit;
    }
}

require_once '../../../includes/db.php';

// ─── ENTRADA ──────────────────────────────────────────────────
$input   = json_decode(file_get_contents('php://input'), true) ?: [];
$sala_id = (int)($input['sala_id'] ?? 0);

if ($sala_id <= 0) {
    http_response_code(400);
    echo json_encode(['status' => 'erro', 'message' => 'sala_id inválido']);
    exit;
}

$user_id = isset($input['user_id']) ? (int)$input['user_id'] : (int)($_SESSION['usuario_id'] ?? 0);

// ─── CONFIG: ritmo do sorteio (ms) ────────────────────────────
// Você pode buscar isso do banco também, mas aqui deixei seguro e simples.
// Se quiser, substitui por SELECT em configuracoes_jogo.
$DEFAULT_INTERVAL_MS = 5000; // 1s (igual seu velocidadeSorteio padrão)
$MIN_INTERVAL_MS     = 250;  // trava pra não ficar spam
$MAX_INTERVAL_MS     = 5000; // trava teto

// ─── LOG ──────────────────────────────────────────────────────
$logDir = __DIR__ . '/../logs';
if (!is_dir($logDir)) @mkdir($logDir, 0775, true);
function bingoLog(string $msg): void {
    global $logDir;
    @error_log("[".date('Y-m-d H:i:s')."] $msg\n", 3, $logDir . '/bingo_debug.log');
}

// ─── AJUDA: pega intervalo do banco (opcional) ────────────────
function getIntervalMs(PDO $pdo, int $default, int $min, int $max): int {
    try {
        $stmt = $pdo->query("SELECT velocidade_sorteio FROM configuracoes_jogo WHERE ativo=1 LIMIT 1");
        $v = $stmt ? $stmt->fetchColumn() : null;
        $sec = is_numeric($v) ? (int)$v : null;
        if ($sec === null || $sec <= 0) return $default;
        $ms = $sec * 1000;
        return max($min, min($max, $ms));
    } catch (Throwable $e) {
        return $default;
    }
}

// ─── PROCESSAMENTO ────────────────────────────────────────────
try {
    $interval_ms = getIntervalMs($pdo, $DEFAULT_INTERVAL_MS, $MIN_INTERVAL_MS, $MAX_INTERVAL_MS);

    $pdo->beginTransaction();

    // Lock da sala
    $stmt = $pdo->prepare("SELECT * FROM salas_bingo WHERE id = ? FOR UPDATE");
    $stmt->execute([$sala_id]);
    $sala = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$sala) {
        $pdo->rollBack();
        echo json_encode(['status' => 'erro', 'message' => 'Sala não encontrada']);
        exit;
    }

    $numeros_sorteados = json_decode($sala['numeros_sorteados'] ?? '[]', true) ?: [];
    $total_bolas = count($numeros_sorteados);

    // ─── SALA NÃO INICIOU ────────────────────────────────────
    if ($sala['status'] === 'aguardando') {
        $pdo->rollBack();
        echo json_encode([
            'status'            => 'aguardando',
            'total_bolas'       => 0,
            'numeros_sorteados' => []
        ]);
        exit;
    }

    // ─── SALA FINALIZADA/FECHADA ────────────────────────────
    if ($sala['status'] === 'finalizado' || $sala['status'] === 'bingo_fechado') {
        $pdo->rollBack();
        echo json_encode([
            'status'            => $sala['status'],
            'total_bolas'       => $total_bolas,
            'numeros_sorteados' => $numeros_sorteados
        ]);
        exit;
    }

    // ─── TRATAMENTO DE BINGO_FEITO ──────────────────────────
    if (!empty($input['bingo_feito']) && $input['bingo_feito'] === true) {
        if ($sala['status'] === 'em_andamento') {
            $upd = $pdo->prepare("UPDATE salas_bingo SET status='bingo_fechado', fim_jogo=NOW() WHERE id=?");
            $upd->execute([$sala_id]);
            $pdo->commit();

            bingoLog("BINGO FECHADO — sala #{$sala_id} pelo usuario {$user_id}");

            echo json_encode([
                'status'            => 'bingo_fechado',
                'total_bolas'       => $total_bolas,
                'numeros_sorteados' => $numeros_sorteados,
                'next_in_ms'        => $interval_ms
            ]);
            exit;
        }

        $pdo->rollBack();
        echo json_encode([
            'status'            => $sala['status'],
            'total_bolas'       => $total_bolas,
            'numeros_sorteados' => $numeros_sorteados
        ]);
        exit;
    }

    // ─── SE JÁ TEM 75 → FINALIZA ────────────────────────────
    if ($total_bolas >= 75) {
        $pdo->prepare("UPDATE salas_bingo SET status='finalizado', fim_jogo=NOW() WHERE id=?")->execute([$sala_id]);
        $pdo->commit();

        echo json_encode([
            'status'            => 'finalizado',
            'total_bolas'       => 75,
            'numeros_sorteados' => $numeros_sorteados,
            'next_in_ms'        => $interval_ms
        ]);
        exit;
    }

    // ─── CONTROLE DE TEMPO ENTRE BOLAS (SERVIDOR) ────────────
    // Você precisa ter a coluna `proxima_bola_em` em salas_bingo (DATETIME NULL).
    // Se não tiver, crie:
    // ALTER TABLE salas_bingo ADD COLUMN proxima_bola_em DATETIME NULL;
    //
    // Regras:
    // - Se proxima_bola_em está no futuro → retorna aguardando_bola com wait_ms
    // - Se está vazio/atrasado → libera nova bola e seta proxima_bola_em = NOW() + intervalo
    $proxima_em = $sala['proxima_bola_em'] ?? null;
    $nowTs = time();

    if (!empty($proxima_em)) {
        $proximaTs = strtotime($proxima_em);
        if ($proximaTs !== false && $proximaTs > $nowTs) {
            $wait_ms = ($proximaTs - $nowTs) * 1000;
            $pdo->rollBack();
            echo json_encode([
                'status'            => 'aguardando_bola',
                'wait_ms'           => max(120, (int)$wait_ms),
                'next_in_ms'        => max(120, (int)$wait_ms),
                'total_bolas'       => $total_bolas,
                'numeros_sorteados' => $numeros_sorteados
            ]);
            exit;
        }
    }

    // ─── GERA A PRÓXIMA BOLA ─────────────────────────────────
    $todas = range(1, 75);
    $disponiveis = array_values(array_diff($todas, $numeros_sorteados));

    if (empty($disponiveis)) {
        $pdo->prepare("UPDATE salas_bingo SET status='finalizado', fim_jogo=NOW() WHERE id=?")->execute([$sala_id]);
        $pdo->commit();

        echo json_encode([
            'status'            => 'finalizado',
            'total_bolas'       => $total_bolas,
            'numeros_sorteados' => $numeros_sorteados,
            'next_in_ms'        => $interval_ms
        ]);
        exit;
    }

    // Escolhe uma bola aleatória
    $nova_bola = $disponiveis[array_rand($disponiveis)];
    $numeros_sorteados[] = (int)$nova_bola;
    $total_bolas = count($numeros_sorteados);

    // Define próxima liberação
    $nextTs = $nowTs + (int)ceil($interval_ms / 1000);
    $nextDt = date('Y-m-d H:i:s', $nextTs);

    // Persiste
    $pdo->prepare("UPDATE salas_bingo SET numeros_sorteados=?, proxima_bola_em=? WHERE id=?")
        ->execute([json_encode($numeros_sorteados), $nextDt, $sala_id]);

    // Se bateu 75 nessa bola, finaliza
    if ($total_bolas >= 75) {
        $pdo->prepare("UPDATE salas_bingo SET status='finalizado', fim_jogo=NOW() WHERE id=?")->execute([$sala_id]);
    }

    $pdo->commit();

    echo json_encode([
        'status'            => ($total_bolas >= 75 ? 'finalizado' : 'ok'),
        'numero'            => (int)$nova_bola,
        'total_bolas'       => $total_bolas,
        'numeros_sorteados' => $numeros_sorteados,
        'next_in_ms'        => $interval_ms
    ]);
    exit;

} catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    bingoLog("ERRO sorteio_server.php: " . $e->getMessage());

    http_response_code(500);
    echo json_encode(['status' => 'erro', 'message' => 'Erro interno']);
    exit;
}
